#ifndef _XMLREADER_H_
#define _XMLREADER_H_

#include <boost/filesystem/path.hpp>
#include <string>
#include <stdexcept>

#include <outilex/xml.h>
#include <libxml/xmlreader.h>


class xmlreader {

public:

  xmlreader() : reader(NULL), status() {}

  xmlreader(const boost::filesystem::path & path) : reader(NULL), status() { open(path); }
 
  ~xmlreader() { close(); }

  void open(const boost::filesystem::path &path) {
    close();
    std::string fname = path.native_file_string();
    reader = xmlReaderForFile(fname.c_str(), NULL, 0);
    if (! reader) { throw std::runtime_error("xmlreader: cannot open " + fname); }
    xmlTextReaderSetErrorHandler(reader, xml_reader_default_error_handler, & status);
  }

  void close() {
    if (reader) {
      xmlFreeTextReader(reader);
      status.clear();
      reader = NULL;
    }
  }

  void verbose_error(bool v) {
    if (v) {
      xmlTextReaderSetErrorHandler(reader, xml_reader_default_error_handler, & status);
    } else {
      xmlTextReaderSetErrorHandler(reader, xml_reader_noverbose_error_handler, & status);
    }
  }


  void check() { xmlcheck(status); }

  operator bool() const { return reader && status.severity == 0; }
  //bool operator!() const { return ! *this; }


  int get_parser_prop(int prop) { return xmlTextReaderGetParserProp(reader, prop); }

  int set_parser_prop(int prop, bool active) {
    return xmlTextReaderSetParserProp(reader, prop, active);
  }
  
  int read_state() { return xmlTextReaderReadState(reader); }
  int node_type() { return xmlTextReaderNodeType(reader); }

  int read() { return xmlTextReaderRead(reader); }
  int next() { return xmlTextReaderNext(reader); }

  xmlNodePtr expand() { return xmlTextReaderExpand(reader); }

  int is_valid() { return xmlTextReaderIsValid(reader); }	

  int is_empty_element() { return xmlTextReaderIsEmptyElement(reader); }

  char * name() { return (char *) xmlTextReaderName(reader); }

  char * local_name() { return (char *) xmlTextReaderLocalName(reader); }
  const char * const_name() { return (const char *) xmlTextReaderConstName(reader); }

  int has_value() { return xmlTextReaderHasValue(reader); }

  char * const_value() { return (char *) xmlTextReaderConstValue(reader); }

  int depth() { return xmlTextReaderDepth(reader); }


  char * value() { return (char *) xmlTextReaderValue(reader); }
  char * read_string() { return (char *) xmlTextReaderReadString(reader); }


  int move_to_element() { return xmlTextReaderMoveToElement(reader); }

  /* attributes */

  int has_attributes() { return xmlTextReaderHasAttributes(reader); }
  int attribute_count() { return xmlTextReaderAttributeCount(reader); }

  int move_to_attribute(const std::string & name) {
    return xmlTextReaderMoveToAttribute(reader, (xmlChar *) name.c_str());
  }

  int move_to_attribute_no(int no) { return xmlTextReaderMoveToAttributeNo(reader, no); }

  int move_to_first_attribute() { return xmlTextReaderMoveToFirstAttribute(reader); }

  int move_to_next_attribute() { return xmlTextReaderMoveToNextAttribute(reader); }

  char * get_attribute(const std::string & name) {
    return (char *) xmlTextReaderGetAttribute(reader, (xmlChar *) name.c_str());
  }

  char * get_attribute_no(int no) {
    return (char *) xmlTextReaderGetAttributeNo(reader, no);
  }


protected:
  xmlTextReaderPtr reader;
  xml_status status;
};

#endif
