#include "table.h"
#include "utilities.h"
#include <stdlib.h>
#include <assert.h>

//Uchar FIELD_SEPARATOR = (Uchar) '\t';
Uchar FIELD_SEPARATOR = (Uchar) ',';


int table_n_free = 0;
int table_n_malloc = 0;

void* table_malloc(int n){
  table_n_malloc++;
  return (void *)malloc(n);
}

void table_pfree(void *ptr){
  if(ptr == NULL) return;
  table_n_free++;
  free(ptr);
  ptr = NULL;
}

int table_print_n_malloc(){
  printf("lex_malloc=%d\n",table_n_malloc);
  return table_n_malloc;
}

int table_print_n_free(){
  printf("lex_free=%d\n",table_n_free);
  return table_n_free;
}

void table_print(Table tab){
  if(tab == NULL) {
    printf("NULL TABLE\n");
    return;
  }
  printf("HEADER\n");
  lex_print(tab->header);
  printf("LEXICON\n");
  lex_print(tab->lex);
  printf("DEFAULT HEADER\n");
  tab_i_println(tab->default_header);
  printf("CONTENT");
  tab2_i_println(tab->content);
}


Table table_init(){
  Table tab = NULL;
  tab = (Table) table_malloc(sizeof(struct table));
  if(tab == NULL) util_error("table_init","allocation error");

  tab->n_rows = 0;
  tab->n_columns = 0;
  tab->content = NULL;
  tab->lex = lex_init();
  tab->header = lex_init();
  tab->default_header = NULL;
  lex_add_string_element(&(tab->lex),"-");
  lex_add_string_element(&(tab->lex),"+");
  lex_add_string_element(&(tab->lex),"<E>");
  lex_add_string_element(&(tab->lex),"~");
  tab->content = NULL;
  return tab;
}

int table_set_epsilon_as_default(char *config,int header_type){
  if((header_type == TABLE_TWO_ROW_HEADER) && (config != NULL)){
    util_error("table_load","I there is a config file, there should be only one row for header");
  }
  if(config != NULL) return 0;
  if(header_type == TABLE_TWO_ROW_HEADER) return 0;
  return 1;
}

int table_get_header(Table *tab, FILE *f, int encoding){

  Uchar EMPTY[] = { 'E', 'M', 'P', 'T', 'Y', '0', 0 };

  Ustring line = NULL,u;
  Tab_u temp = NULL;
  int n_col,i;
  char element;

  line = ustring_get_line(f,encoding);
  if((line == NULL) || (ustring_length(line) == 0)) util_error("table_get_header","Missing header row");
  temp = ustring_split(FIELD_SEPARATOR,line);
  //  tab_u_print(temp);
  n_col = tab_u_get_n_elements(temp);

  //printf("get_header:\n");

  for(i = 0 ; i < n_col ; i++){
    u = tab_u_get_value(temp,i);

    if (*u == 0) { // if no title (empty colomn), make a uniq id
      EMPTY[5]++;
      u = EMPTY;
    }

    lex_add_element2(&((*tab)->header),u,&element);
    if(element == LEX_EXISTING_ELEMENT){
      ustring_println(u);
      printf("hey!!!!!!\n");
      util_error("table_get_header","Column title already exists");
    }
  }
  return n_col;
}

int table_get_row(Table *t, FILE *f, int default_header, int encoding, int row){

  Tab_u temp = NULL;
  Ustring line = NULL;
  int n_col,col;

  line = ustring_get_line(f,encoding);
  if((line == NULL) || (ustring_length(line) == 0)) {
    if(default_header) util_error("table_get_row","Missing default value");
    return 0;
  }

  temp = ustring_split(FIELD_SEPARATOR,line);
  n_col = tab_u_get_n_elements(temp);

  if((*t)->n_columns != n_col){
    fprintf(stderr, "error: table_get_row: got %d colomns instead of %d\n", n_col, (*t)->n_columns);
    fprintf(stderr, "dump of row:\n");
    for (col = 0; col < n_col; ++col) {
      Ustring value = tab_u_get_value(temp, col);
      fprintf(stderr, "%d: ", col);
      ustring_write(stderr, value);
      fputc('\n', stderr);
    }
 
    util_error("table_get_row", "Missing elements in row");
  }

  for(col = 0 ; col < n_col ; col++){

    Ustring value = tab_u_get_value(temp, col);
    int idx = lex_add_element(&((*t)->lex), value);

    //val = lex_add_element(&((*t)->lex),tab_u_get_value(temp,col));

    if(default_header) {

      if (*value) {
        tab_i_assign_value(&((*t)->default_header),col,idx);
      } else { // empty string, replace by <E>
        tab_i_assign_value(&((*t)->default_header),col,TABLE_EPSILON);
      }
    }
    else tab2_i_assign_value(&((*t)->content),row,col,idx);
  }
  return 1;
}

void table_load_content(char *name,Table *tab, int header_type,int encoding){

  FILE *f;
  int DEFAULT_HEADER = 1,NORMAL_ROW = 0,i = 0;
 

  f = fopen(name,"r");
  if(f == NULL) util_error("table_load_content","Cannot open table");
  
  (*tab)->n_columns = table_get_header(tab,f,encoding);
  if(header_type == TABLE_TWO_ROW_HEADER){
    table_get_row(tab,f,DEFAULT_HEADER, encoding,i);
  }

  //printf("i'm here: %d properties\n", (*tab)->n_columns);

  i++;
  while (table_get_row(tab,f,NORMAL_ROW, encoding,i)) {
    //printf("row %d\n", (*tab)->n_rows);
    ((*tab)->n_rows)++;
    i++;
  }

  printf("table loaded (%d properties, %d rows)\n", (*tab)->n_columns, i);
  //printf("table loaded (%d rows)\n", i);
  fclose(f);
}

void table_load_config( char *name,Table *tab,int n_col,int encoding){
  FILE *f = fopen(name,"r");
  Ustring u;
  Tab_u temp;
  int i,val_index;

  if(f == NULL) util_error("table_load_config","cannot open file");
  u = ustring_get_line(f,encoding);
   if(u == NULL)util_error("table_load_config","problem with reading first row");
  temp = ustring_split(FIELD_SEPARATOR,u);
  if(n_col != tab_u_get_n_elements(temp)) util_error("table_load_config","number of columns (first row)");
  ustring_free(u);
  tab_u_free(&temp);
  u = ustring_get_line(f,encoding);
  if(u == NULL)util_error("table_load_config","problem with reading second row");
  temp = ustring_split(FIELD_SEPARATOR,u);
  if(n_col != tab_u_get_n_elements(temp)) util_error("table_load_config","number of columns (second row)");
  for(i = 0 ; i < n_col ; i++){
    val_index = lex_add_element(&((*tab)->lex),tab_u_get_value(temp,i));
    tab_i_assign_value(&((*tab)->default_header),i,val_index);
  }
  ustring_free(u);
  tab_u_free(&temp);
  fclose(f);
}

void table_init_default_to_epsilon(Table *tab,int epsilon){
  int i,n = table_get_n_columns(*tab);
  for(i = 0 ; i < n ; i++){
    tab_i_assign_value(&((*tab)->default_header),i,epsilon);    
  }
}

int table_get_parameter_type(Ustring parameter){
  switch(ustring_get_uchar(parameter,0)){
  case '!':
    return TABLE_NOT_PARAMETER;
  case '$':
    return TABLE_PREFIX_PARAMETER;
  case '%':
    return TABLE_ROWID_PARAMETER;
  case ':':
    return TABLE_DEFAULT_PARAMETER;
   default: 
    return TABLE_NORMAL_PARAMETER;
  }
}

/* param type is NOT, DEFAULT or NORMAL
 */
Ustring table_resolve_elementary_parameter(Table tab,int row, Ustring parameter, int type) {

  Uchar epsilon[] = { '<', 'E', '>', 0 };

  Ustring value, default_value;

  value = table_get_value(tab, row, parameter);
  default_value = table_get_default_value(tab,parameter);

  if (value == NULL) return NULL;

  int idx = lex_get_index(tab->lex, value);

  if (type == TABLE_NOT_PARAMETER) {
    if (idx == TABLE_MINUS) { return ustring_allocation(epsilon); }
    return NULL;
  }

  if (type == TABLE_DEFAULT_PARAMETER) {
    if (idx == TABLE_MINUS) { return NULL; }
    return ustring_allocation(default_value);
  }

  // NORMAL parameter
 
  assert(type == TABLE_NORMAL_PARAMETER);

  if (idx == TABLE_MINUS) { return NULL; }
 
  // tilde is synonym for plus
  if ((idx == TABLE_PLUS) || (idx == TABLE_TILDE)) { return ustring_allocation(epsilon); }
  return ustring_allocation(value);

/*
  if (lex_get_index(tab->lex,value) == TABLE_MINUS) {
    //if (type == TABLE_NOT_PARAMETER) return ustring_allocation(default_value);
    if (type == TABLE_NOT_PARAMETER) return ustring_allocation(epsilon);
    return NULL;
  }

  // TILDE is synonym for PLUS
  if ((lex_get_index(tab->lex,value) == TABLE_PLUS) || lex_get_index(tab->lex, value) == TABLE_TILDE) { 
    if (type == TABLE_NOT_PARAMETER) return NULL;
    if (type == TABLE_DEFAULT_PARAMETER) return ustring_allocation(default_value);
    return ustring_allocation(epsilon);
  }

  return ustring_allocation(value);  
*/
}

Ustring table_resolve_prefix_parameter(Table tab,int row, Ustring parameter){

  Ustring param,temp,u = NULL,v;
  Llist_i cols = NULL;
  struct ll_i_cell *ptr;

  param = ustring_get_substr(parameter,1,ustring_length(parameter));
  //temp = ustring_from_string(" =:");
  temp = ustring_from_string("=");
  param = ustring_strcat(&param,temp);
  ustring_free(temp);
  cols = lex_find_prefixed_elements(tab->header,param); //get all headers staring with prefix param (e.g. N1 =:)
  ustring_free(param);

  if (cols) {
    ptr = cols->head;
  } else {

    param = ustring_get_substr(parameter,1,ustring_length(parameter));
    temp = ustring_from_string(" =:");
    param = ustring_strcat(&param,temp);
    ustring_free(temp);
    cols = lex_find_prefixed_elements(tab->header,param); //get all headers staring with prefix param (e.g. N1 =:)
    ustring_free(param);

    if (cols) {
      ptr = cols->head;
    
    } else {
      printf("error: resolve prefix: '");
      ustring_print(parameter);
      printf("' not found\n");
      ptr = NULL;
    }
  }
  

  while(ptr != NULL){    //for each header column prefixed by param
    param = lex_get_element(tab->header,ptr->value);
    v = table_resolve_elementary_parameter(tab,row, param, TABLE_DEFAULT_PARAMETER);
    if(v != NULL){
      if(u != NULL){
	temp = ustring_from_string("+");
	ustring_strcat(&temp,v);
      }
      else{
	temp = ustring_allocation(v);
      }
      ustring_free(v);
      ustring_strcat(&u,temp);
    }
    ptr = ptr->next;    
  }
  ll_i_free(&cols);

  return u;
}
Ustring table_resolve_meta_parameter(Table tab,int row, Ustring parameter){
  Ustring meta = ustring_get_substr(parameter,1,ustring_length(parameter));
  return table_resolve_parameter(tab,row, meta);
}

Ustring table_resolve_parameter(Table tab,int row, Ustring parameter){
  
  int parameter_type = table_get_parameter_type(parameter);
  
  if (parameter_type == TABLE_PREFIX_PARAMETER){
    return table_resolve_prefix_parameter(tab,row,parameter);
  }
  
  if (parameter_type == TABLE_ROWID_PARAMETER) {
    char buf[64];
    sprintf(buf, "%d", row);
    return ustring_from_string(buf);
  }

  if (parameter_type != TABLE_NORMAL_PARAMETER) { // skip the ':', '!', etc.  prefix
    parameter += 1;
  }

  return table_resolve_elementary_parameter(tab,row,parameter,parameter_type);
}

Ustring table_get_default_value(Table tab,Ustring col){
   int col_index = lex_get_index(tab->header,col);
   if(col_index == -1){
     //ustring_println(col);
     //     util_warning("table_get_default_value","column header does not exist");
     return NULL;
   }
   int val_index = tab_i_get_value(tab->default_header,col_index);
   return lex_get_element(tab->lex,val_index);
}

Ustring table_get_value_with_indexes(Table tab,int row,int col){
  int val_index = tab2_i_get_value(tab->content,row,col);
  return lex_get_element(tab->lex,val_index);
}

Ustring table_get_value(Table tab,int row,Ustring col, bool warnifnotexist){

  int col_index = lex_get_index(tab->header,col);

  if (col_index == -1) {
    if (warnifnotexist) {
      ustring_println(col, stderr);
      util_warning("table_get_value","column header does not exist");
    }
    return NULL;
  }
  return table_get_value_with_indexes(tab,row,col_index);
}


Table table_load(char name[],char configname[],int encoding, int header_type){

  Table tab = table_init();
  int use_epsilon_as_default = table_set_epsilon_as_default(configname, header_type);

  table_load_content(name,&tab,header_type,encoding);

  if (configname != NULL) table_load_config(configname,&tab,tab->n_columns,encoding);  
  if (use_epsilon_as_default) table_init_default_to_epsilon(&tab,TABLE_EPSILON);

  return tab;
}

int table_get_n_rows(Table tab){
  return tab->n_rows;
}

int table_get_n_columns(Table tab){
  return tab->n_columns;
}

void table_free(Table *tab){
  lex_free(&((*tab)->lex));
  lex_free(&((*tab)->header));
  tab_i_free(&((*tab)->default_header));
  tab2_i_free(&((*tab)->content));

  table_pfree(*tab);
}
