#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>



#include "table.h"
#include "utilities.h"


bool utf8_output = false;

FILE * utf8_open(char * name) {
  char cmd[strlen(name) + 64];
  sprintf(cmd, "recode utf16le..utf8 > %s8", name);
//  fprintf(stderr, "name=%s, cmd=%s\n", name, cmd);
  return popen(cmd, "w"); // add '8' suffix to grf
}


Tab_u comp_load_reference_grf(char *name,int encoding){
  Tab_u temp = NULL;
  Ustring line;
  FILE *f;
  int i = 0;

  f = fopen(name, "r");
  if(f == NULL) util_error("comp_load_reference_grf","cannot open reference graph");

  fgetc(f);
  fgetc(f);
  while((line = ustring_get_line(f,encoding)) != NULL){
    tab_u_assign_value(&temp,i,line);
    ustring_free(line);
    i++;
  }  
  return temp;
}

int comp_is_box(Ustring line){
  return (ustring_get_uchar(line,0) == (Uchar)'"');
}

int comp_is_parameter(int index){
  return (index % 2);
}


Ustring comp_process_parameter(Table tab, int row, Ustring parameter) {
  return table_resolve_parameter(tab,row,parameter);
}

#if 0
Ustring comp_fill_box_content(Table tab,int row,Ustring box){
  Ustring u = NULL,val,v;
  int i;
  Tab_u split;

  split = ustring_split((Uchar)'@',box);
  
  for(i = 0 ; i < tab_u_get_n_elements(split); i++){
    val = tab_u_get_value(split,i);
    if(comp_is_parameter(i)){
      v = comp_process_parameter(tab,row,val);
      if(v == NULL){
	ustring_free(u);
	tab_u_free(&split);
	return NULL;
      }
      ustring_strcat(&u,v);
      ustring_free(v);
    }
    else{
      ustring_strcat(&u,val);
    }
  }
  tab_u_free(&split);
  return u;
}
#endif

Ustring comp_fill_box_content2(Table tab,int row, Ustring box, Ustring SFX){

  Uchar suffix[] = { 's', 'f', 'x', 0 };

  Ustring res = NULL;

  Uchar * label = ustring_copy(box + 1);

  int labelend = ustring_find_last(label, '"');
  label[labelend] = 0;

  Uchar * input  = label;
  Uchar * output = NULL;
  int slash = ustring_find(label, '/');

  if (slash != -1) {
    label[slash] = 0;
    output = label + slash + 1;
  }

  // first split input by row (separated by '+')

  Tab_u rows = ustring_split('+', input);

  ustring_strcat(& res, '"');

  for (int r = 0; r < tab_u_get_n_elements(rows); ++r) {

    if (r > 0) { ustring_strcat(& res, '+'); }

    Ustring currow = tab_u_get_value(rows, r);
    Tab_u split = ustring_split('@', currow);

    for (int i = 0 ; i < tab_u_get_n_elements(split); i++) {

      Ustring val = tab_u_get_value(split,i);

      if (comp_is_parameter(i)) {

        if (ustring_cmp(val, suffix) == 0) {

          ustring_strcat(& res, SFX);

        } else {

          Ustring v = comp_process_parameter(tab,row,val);
        
          if (v == NULL) {
            ustring_free(res);
            tab_u_free(&split);
            return NULL;
          }
          ustring_strcat(&res,v);
          ustring_free(v);
        }

      } else { ustring_strcat(&res,val); }
    }
    tab_u_free(&split);

    if (currow[0] == ':') { // subgraph call
      int len = ustring_length(res);
      if (res[len - 1] == '$') { // replace ending '$' by SFX
        res[len - 1] = 0;
        ustring_strcat(& res, SFX);
      }
    }
  }

  tab_u_free(& rows);

  if (output) {

    ustring_strcat(& res, '/');

    Tab_u split = ustring_split('@', output);

    for (int i = 0 ; i < tab_u_get_n_elements(split); i++) {

      Ustring val = tab_u_get_value(split, i);

      if (comp_is_parameter(i)) {

        if (ustring_cmp(val, suffix) == 0) {

          ustring_strcat(& res, SFX);

        } else {

          Ustring v = comp_process_parameter(tab,row,val);
        
          if (v == NULL) {
            ustring_free(res);
            tab_u_free(&split);
            return NULL;
          }
          ustring_strcat(&res,v);
          ustring_free(v);
        }

      } else { ustring_strcat(&res,val); }
    }
    tab_u_free(&split);
  }

  label[labelend] = '"';
  ustring_strcat(&res, label + labelend);

  ustring_free(label);
  return res;
}


void comp_get_box_position(Ustring box,int *x,int *y){
  int pos = ustring_find_last(box,(Uchar)'"');
  if(pos == -1) util_error("comp_get_box_position","cannot find \" in the box line");
  *x = ustring_get_next_int(box,&pos);
  *y = ustring_get_next_int(box,&pos);
}


void comp_process_box(FILE * f, Table tab, int row, Ustring line, Ustring SFX){

  Ustring box,ufail;
  int x,y,max = 10000;
  char fail[max];

  box = comp_fill_box_content2(tab,row,line, SFX);

  if(box != NULL) { //succes
    ustring_writeln_utf_16(f,box);
    ustring_free(box);  
  } else {
    comp_get_box_position(line,&x,&y);
    snprintf(fail,max,"\"<E>\" %d %d 0 ",x,y);
    ufail = ustring_from_string(fail);
    //   ustring_println(ufail);
    ustring_writeln_utf_16(f,ufail);
    ustring_free(ufail);
  }
}


void comp_process_grf(Table tab, int row, char *dest_path, Tab_u grf, Ustring SFX){

  FILE * f;

  if (utf8_output) {
    f = utf8_open(dest_path);
  } else {
    f = fopen(dest_path, "w");
  }

  int i,n = tab_u_get_n_elements(grf);
  Ustring line = NULL;

  // printf("%s\n",dest_path);
  if(f == NULL) util_error("comp_process_grf","cannot open new grf file");

  if (! utf8_output) {
    ustring_write_starting_mark(f); 
  }

  for(i = 0 ; i < n ; i++){ // for each line
    line = tab_u_get_value(grf,i);
    if(comp_is_box(line)){
      comp_process_box(f, tab, row, line, SFX);
    }
    else{
      ustring_writeln_utf_16(f,line);
    }
  }

  if (utf8_output) {
    pclose(f);
  } else {
    fclose(f);
  }
}


void comp_build_main_grf(char *name, Ustring content) {

  Ustring u;
  int i;

  int content_size = ustring_length(content);

  int nbbox = content_size / 5000 + 1;

  //  int f = open(name,O_WRONLY|O_CREAT|O_TRUNC,00644);
  FILE * f;
  
  if (utf8_output) {
    f = utf8_open(name);
  } else {
    f = fopen(name, "w");
    ustring_write_starting_mark(f);
  }
  ustring_writeln_utf_16_from_string(f,"#Unigraph");
  ustring_writeln_utf_16_from_string(f,"SIZE 1188 840");
  ustring_writeln_utf_16_from_string(f,"FONT Times New Roman:  10");
  ustring_writeln_utf_16_from_string(f,"OFONT Times New Roman:B 12");
  ustring_writeln_utf_16_from_string(f,"BCOLOR 16777215");
  ustring_writeln_utf_16_from_string(f,"FCOLOR 0");
  ustring_writeln_utf_16_from_string(f,"ACOLOR 10066329");
  ustring_writeln_utf_16_from_string(f,"SCOLOR 16711680");
  ustring_writeln_utf_16_from_string(f,"CCOLOR 255");
  ustring_writeln_utf_16_from_string(f,"DBOXES y");
  ustring_writeln_utf_16_from_string(f,"DFRAME y");
  ustring_writeln_utf_16_from_string(f,"DDATE y");
  ustring_writeln_utf_16_from_string(f,"DFILE y");
  ustring_writeln_utf_16_from_string(f,"DDIR n");
  ustring_writeln_utf_16_from_string(f,"DRIG n");
  ustring_writeln_utf_16_from_string(f,"DRST n");
  ustring_writeln_utf_16_from_string(f,"FITS 100");
  ustring_writeln_utf_16_from_string(f,"PORIENT L");
  ustring_writeln_utf_16_from_string(f,"#");

  char buf[16];
  sprintf(buf, "%d", nbbox + 2);
  ustring_writeln_utf_16_from_string(f,buf);

  ustring_write_utf_16_from_string(f,"\"<E>\" 69 199 ");
  sprintf(buf, "%d ", nbbox);
  ustring_write_utf_16_from_string(f,buf);
  for (i = 0; i < nbbox; i++) {
    sprintf(buf, "%d ", i + 2);
    ustring_write_utf_16_from_string(f,buf);
  }
  ustring_writeln_utf_16_from_string(f,"");

  ustring_writeln_utf_16_from_string(f,"\"\" 466 199 0 ");

  int pos = 0;

  for (i = 0; i < nbbox && pos < content_size; ++i) {  

    int end = pos + 5000;
    if (end > content_size) { end = content_size; }

    while (content[end] && content[end] != '+') { end++; }


    u = ustring_from_string("\"");
    ustring_write_utf_16(f,u);
    ustring_free(u);
  
    u = ustring_get_substr(content, pos, end); 
    ustring_write_utf_16(f, u);
    ustring_free(u);
    
    u = ustring_from_string("\" 215 199 1 1 ");
    ustring_writeln_utf_16(f,u);
    ustring_free(u);
    pos = end + 1;
  }

  while (i < nbbox) {
    ustring_writeln_utf_16_from_string(f,"\"<E>\" 69 199 0 ");
    ++i;
  }
  
  if (utf8_output) {
    pclose(f);
  } else {
    fclose(f);
  }
}

void remove_diacritics(char * text) {

  while (*text) {
  
    switch (*text) {
    case '':
    case '':
    case '':
      *text = 'a';
      break;
    case '':
    case '':
    case '':
    case '':
      *text = 'e';
      break;
    case '':
    case '':
      *text = 'i';
      break;
    case '':
    case '':
      *text = 'o';
      break;
    case '':
    case '':
      *text = 'u';
      break;
      //suppress space too
    case ' ':
    case '\t':
      *text = '_';
    }
    ++text;
  }
}


Ustring comp_build_suffix(Table tab, int row, char * suffix) {

  char buf[1024];

  Ustring param = ustring_from_string(suffix);
  Ustring value = table_get_value(tab, row, param);
  free(param);

  if (value)  { 
    char * val2 = ustring_to_string(value);
    sprintf(buf, "-%s-%d", val2, row);
    free(val2);
  } else {
    sprintf(buf, "-%s-%d", suffix, row);
  }

  remove_diacritics(buf);
  Ustring res = ustring_from_string(buf);
  return res;
}


#if 0
void comp_build_name_grf(Table tab,char name[],char *suffix,char dir[], char * basename, int row,int max){

  Ustring param,val;
  char *value;

  if (suffix != NULL) {
    param = ustring_from_string(suffix);
    val = table_get_value(tab,row,param);

    if (val == NULL) { // if suffix is not a table title, put it litteraly
      value = suffix;
    } else {
      value = ustring_to_string(val);
    }
    snprintf (name, max,"%s/%s-%s-%d.grf",dir, basename, value,row);
    ustring_pfree(value);
    remove_diacritics(name);
    return;
  }
  snprintf (name, max,"%s/%s-%d.grf",dir, basename, row);
  remove_diacritics(name);
}
#endif


bool forget(Table tab, int entry) {

  Uchar FORGET[] = { 'F', 'O', 'R', 'G', 'E', 'T', 0 };

  Ustring value = table_get_value(tab, entry, FORGET, false);

  if (value && value[0] == 'X') { return true; }
  return false;
}



Ustring comp_process_ref(Table tab,char *ref,char *dir,char* basename, char * suffix, int encoding) {

  int i,n = table_get_n_rows(tab),max = 500000;
  char name_grf[max],grf_content[max],short_name[max];
  
  grf_content[0] = 0;
  Tab_u temp = comp_load_reference_grf(ref,encoding);
 
  for (i = 1 ; i <= n ; i++) { 

    if (forget(tab, i)) { continue; }

    Ustring SFX = comp_build_suffix(tab, i, suffix);
    char * sfx = ustring_to_string(SFX);

    sprintf(name_grf, "%s/%s%s.grf", dir, basename, sfx);
    free(sfx);

    //printf("%s\n",name_grf);

    comp_process_grf(tab, i, name_grf, temp, SFX);

    if (grf_content[0]) strcat(grf_content, "+");

    strcat(grf_content,":");
    util_get_filename(name_grf,short_name);
    // remove .grf
    short_name[strlen(short_name) - 4] = 0;
    //printf("shortname=%s\n", short_name);
    strcat(grf_content,short_name);    

    ustring_free(SFX);
  }
  tab_u_free(&temp);
  return ustring_from_string(grf_content);
}


void comp_process(Table tab,char *main_grf,char *ref,char *dir,char* basename, char * suffix,
                  int encoding) {
 
  char buf[1024];
  sprintf(buf, "%s/%s.grf", dir, main_grf);
  Ustring content = comp_process_ref(tab, ref, dir, basename, suffix, encoding); 
  comp_build_main_grf(buf,content);
}

void comp_list(Table tab,char *main_grf,char *list,char *dir, char * suffix, int encoding){

  FILE * f = fopen(list, "r");
  char buf[1024],path[1024];

  
  if (f ==NULL) {
    util_error("complist", "cannot open file");    
  }
  util_get_path(list,path);
  if(strlen(path)) strcat(path,"/");

  Ustring content = NULL;
  while (fgets(buf, 1024, f)) {

    if (buf[0] == '#') { continue; }

    char ref[1024];
    buf[strlen(buf) - 1] = 0;
    strcpy(ref,path);
    strcat(ref, buf);
    strcat(ref, ".grf");
    Ustring tmp = comp_process_ref(tab, ref, dir, buf, suffix, encoding);
 
    /* printf("tmp="); ustring_println(tmp); */
    if (content) { ustring_append(&content, '+'); }
    ustring_strcat(& content, tmp);
    ustring_free(tmp);
  }
  
  //printf("content="); ustring_println(content);
  
  sprintf(buf, "%s/%s.grf", dir, main_grf);
  comp_build_main_grf(buf,content);
}


char * progname;

void usage() {
  printf("usage: %s -table <table> [-conf <config>] [-one|-two] "
         "-dir <destdir> [-sfx <suffix>] -main <main_grf> [-grf|-list] <reference>\n",
         progname);
  exit(0);
}


int main(int argc, char**argv){

  /*
  int max = 10000,
  char config[max];
  */

  char * tablename = NULL, * config = NULL, * dir = NULL, * suffix = "entry";
  char * maingrf = NULL, * reference = NULL, * reflist = NULL;
  int n_headers = TABLE_TWO_ROW_HEADER;
  Table tab;


  progname = *argv;
  argv++, argc--;

  
  if (argc == 0){
    usage();
    //fprintf(stderr,"usage:\n./compile <table> <config> <number of headers> <reference> <main_grf> <grf_name_prefix> <suffix>\ncd README_compile\n");
    //return 1;
  }

  while (argc) {

    if (strcmp(*argv, "-table") == 0) {

      argv++, argc--;
      if (! argc) { fprintf(stderr, "bad args\n"); exit(1); }
      tablename = *argv;
    
    } else if (strcmp(*argv, "-conf") == 0) {

      argv++, argc--;
      if (! argc) { fprintf(stderr, "bad args\n"); exit(1); }
      config = *argv;
    
    } else if (strcmp(*argv, "-dir") == 0) {

      argv++, argc--;
      if (! argc) { fprintf(stderr, "bad args\n"); exit(1); }
      dir = *argv;
    
    } else if (strcmp(*argv, "-sfx") == 0) {

      argv++, argc--;
      if (! argc) { fprintf(stderr, "bad args\n"); exit(1); }
      suffix = *argv;
    
    } else if (strcmp(*argv, "-main") == 0) {

      argv++, argc--;
      if (! argc) { fprintf(stderr, "bad args\n"); exit(1); }
      maingrf = *argv;
    
    } else if (strcmp(*argv, "-grf") == 0) {

      argv++, argc--;
      if (! argc) { fprintf(stderr, "bad args\n"); exit(1); }
      reference = *argv;
    
    } else if (strcmp(*argv, "-list") == 0) {

      argv++, argc--;
      if (! argc) { fprintf(stderr, "bad args\n"); exit(1); }
      reflist = *argv;
    
    } else if (strcmp(*argv, "-one") == 0) {

      n_headers = TABLE_ONE_ROW_HEADER;

    } else if (strcmp(*argv, "-two") == 0) {

      n_headers = TABLE_TWO_ROW_HEADER;

    } else if (strcmp(*argv, "-u8") == 0 || strcmp(*argv, "-utf8") == 0) {

      utf8_output = true;

    } else if (strcmp(*argv, "-help") == 0) {

      usage();

    } else { reference = *argv; }
  
    argv++, argc--;
  }


  if (tablename == NULL || maingrf == NULL || (reference == NULL && reflist == NULL)) {
    fprintf(stderr, "missing some argument\n");
    exit(1);
  }
  
  //  if (prefix == NULL) { prefix = maingrf;  }

  /*
  if(!strcmp(argv[3],"ONE")) n_headers = TABLE_ONE_ROW_HEADER;
  else n_headers = TABLE_TWO_ROW_HEADER;
  if(!strcmp(argv[2],"NULL")) tab = table_load(argv[1],NULL,ASCII_FORMAT,n_headers);
  else tab = table_load(argv[1],argv[2],ASCII_FORMAT,n_headers);
  */

  printf("loading table...\n");
  tab = table_load(tablename, config, ASCII_FORMAT, n_headers);

  char basename[1024];
  /*
  if(!strcmp(argv[7],"NULL")) comp_process(tab,argv[5],argv[4],argv[6],NULL,UNICODE_FORMAT);
  else comp_process(tab,argv[5],argv[4],argv[6],argv[7],UNICODE_FORMAT);
  */
  printf("processing...\n");
  if (reference) {
    util_get_filename(reference, basename);
    assert(strcmp(basename + strlen(basename) - 4, ".grf") == 0);
    basename[strlen(basename) - 4] = 0;
    comp_process(tab, maingrf, reference, dir, basename, suffix, UNICODE_FORMAT);
  } else {
    comp_list(tab, maingrf, reflist, dir, suffix, UNICODE_FORMAT);
  }

  printf("ok.\n");
  table_free(&tab);
  return 0;
}

