#ifndef _XML_FORMAT_FST_H_
#define _XML_FORMAT_FST_H_

#include <string>
#include <boost/lexical_cast.hpp>

#include <outilex/xml.h>
#include <outilex/xml-names.h>
#include <outilex/void_type.h>


namespace detail {

template<typename Transition, typename XmlFormater>
void xml_write_fst_trans(xmlwriter & writer, const Transition & tr, XmlFormater formater) {
  
  writer.start_element(xmlnames::TRANS_ELEM);
  writer.write_attribute(xmlnames::TO_ATTR, boost::lexical_cast<std::string>(tr.to()));

  writer.start_element(xmlnames::IN_ELEM);
  formater.write_in(writer, tr.in());
  writer.end_element();

  writer.start_element(xmlnames::OUT_ELEM);
  formater.write_out(writer, tr.out());
  writer.end_element();

  writer.end_element();
}

template<typename Fst, typename XmlFormater>
void xml_read_fst_trans(xmlNode * node, Fst & fst, int q, XmlFormater formater) {

  typename Fst::input_type in;
  typename Fst::output_type out;

  int to = boost::lexical_cast<int>(xmlGetConstProp(node, xmlnames::TO_ATTR));

  node = node->children;
  while (node) {
    if (xmlStrcmp(node->name, xmlnames::IN_ELEM) == 0) {
      formater.read_in(node, in);
    } else if (xmlStrcmp(node->name, xmlnames::OUT_ELEM) == 0) {
      formater.read_out(node, out);
    }
    node = node->next;
  }
  fst.add_trans(q, in, out, to);
}

template<typename Fst, typename XmlFormater>
void xml_read_fst_final_outputs(xmlNode * node, Fst & fst, int q, XmlFormater formater) {

  //fst.final_outputs(q).clear();

  typename Fst::output_type out;

  node = node->children;
  while (node) {
    if (xmlStrcmp(node->name, xmlnames::OUT_ELEM) == 0) {
      formater.read_out(node, out);
      fst.final_outputs(q).insert(out);
    }
    node = node->next;
  }
}

template<typename Fst, typename XmlFormater>
void xml_write_fst_final_outputs(xmlwriter & writer, const Fst & fst, int q,
                                 XmlFormater formater) {
  writer.start_element(xmlnames::FINAL_OUTPUTS_ELEM);
  typename Fst::outputs_type::const_iterator it, end;
  for (it = fst.final_outputs_begin(q), end = fst.final_outputs_end(q);
       it != end; ++it) {
  
    writer.start_element(xmlnames::OUT_ELEM);
    formater.write_out(writer, *it);
    writer.end_element();
  }
  writer.end_element();
}


template<typename Fst, typename XmlFormater>
void xml_read_fst_state(xmlNode * node, Fst & fst, int q, XmlFormater formater) {

  char * text = xmlGetConstProp(node, xmlnames::FINAL_ATTR);
  if (text && text[0] == '1') { fst.set_final(q); }

  text = xmlGetConstProp(node, xmlnames::SIZE_ATTR);
  if (text) { fst.trans_reserve(q, boost::lexical_cast<int>(text)); }


  node = node->children;
  while (node) {

    if (xmlStrcmp(node->name, xmlnames::PROP_ELEM) == 0) {

      formater.read_fst_state_data(node->children, fst, q);
 
    } else if (xmlStrcmp(node->name, xmlnames::FINAL_OUTPUTS_ELEM) == 0) {

      xml_read_fst_final_outputs(node, fst, q, formater);

    } else if (xmlStrcmp(node->name, xmlnames::TRANS_ELEM) == 0) {

      xml_read_fst_trans(node, fst, q, formater);
    }

    node = node->next;
  }
}


template<typename Fst, typename XmlFormater>
void xml_write_fst_state(xmlwriter & writer, const Fst & fst, int q, XmlFormater formater) {

  writer.start_element(xmlnames::STATE_ELEM);
  
  if (fst.final(q)) {
    writer.write_attribute(xmlnames::FINAL_ATTR, "1");
    if (! fst.final_outputs(q).empty()) {
      xml_write_fst_final_outputs(writer, fst, q, formater);
    }
  }
 
  writer.write_attribute("id", boost::lexical_cast<std::string>(q));

  if (typeid(fst.fst_data(q)) != typeid(void_type)) {
    writer.start_element(xmlnames::PROP_ELEM);
    formater.write_fst_state_data(writer, fst, q);
    writer.end_element();
  }

  typename Fst::const_trans_iterator tr, end;
  for (tr = fst.trans_begin(q), end = fst.trans_end(q); tr != end; ++tr) {
    xml_write_fst_trans(writer, *tr, formater);
  }

  writer.end_element();
}

} // namespace detail


template<typename Fst>
struct DefaultFstXmlFormater {

  typedef Fst fst_type;
  typedef typename fst_type::input_type input_type;
  typedef typename fst_type::output_type output_type;


  const char * fst_name() const { return xmlnames::FST_ELEM; }

  void read_fst_data(xmlNode * node, fst_type & fst) {}
  void write_fst_data(xmlwriter & writer) {}


  void read_fst_state_data(xmlNode * node, fst_type & fst, int q) {
    xml_read(node, fst.fst_data(q));
  }

  void write_fst_state_data(xmlwriter & writer, const fst_type & fst, int q) {
    xml_write(writer, fst.fst_data(q));
  }


  void read_in(xmlNode * node, input_type & in) { xml_read(node, in); }
  void write_out(xmlwriter & writer, const output_type & out) { xml_write(writer, out); }

  void read_out(xmlNode * node, output_type & out) { xml_read(node, out); }
  void write_in(xmlwriter & writer, const input_type & in) { xml_write(writer, in); }
};


template<typename Fst, typename XmlFormater>
void xml_read_fst(xmlNode * node, Fst & fst, XmlFormater formater) {

  fst.clear();

  char * text = xmlGetConstProp(node, xmlnames::SIZE_ATTR);
  if (text) {
    fst.reserve(boost::lexical_cast<int>(text));
  }

  int qno = 0;
  node = node->children;
  while (node) {
    
    if (xmlStrcmp(node->name, xmlnames::STATE_ELEM) == 0) {
    
      fst.resize(qno + 1);
      detail::xml_read_fst_state(node, fst, qno, formater);
      ++qno;
    
    } else if (xmlStrcmp(node->name, xmlnames::PROP_ELEM) == 0) {

      formater.read_fst_data(node, fst);
    }

    node = node->next;
  }
}

template<typename Fst>
void xml_read_fst(xmlNode * node, Fst & fst) {
  xml_read_fst(node, fst, DefaultFstXmlFormater<Fst>());
}


template<typename Fst, typename XmlFormater>
void xml_read_fst(const boost::filesystem::path & path, Fst & fst,
                  XmlFormater formater) {

  std::string fname = path.native_file_string();
  
  xmlDocPtr doc = xmlParseFile(fname.c_str());

  if (doc == NULL) {
    throw std::runtime_error("xml_read_fst: " + fname + " is not a well formed xml document");
  }

  xmlNode * node = xmlDocGetRootElement(doc);

  if (xmlStrcmp(node->name, formater.fst_name()) != 0) {
    throw std::runtime_error("xml_read_fst: " +  fname + ": wrong document type"); 
  }
  try {
    xml_read_fst(node, fst, formater);
  } catch (std::exception & e) {
    xmlFreeDoc(doc);
    throw;
  }
}

template<typename Fst>
void xml_read_fst(const boost::filesystem::path & path, Fst & fst) {
  xml_read_fst(path, fst, DefaultFstXmlFormater<Fst>());
}


template<typename Fst, typename XmlFormater>
void xml_write_fst(xmlwriter & writer, const Fst & fst, XmlFormater formater) {

  int size = fst.size();

  writer.start_element(formater.fst_name());
  writer.write_attribute(xmlnames::SIZE_ATTR, boost::lexical_cast<std::string>(size));

  writer.start_element(xmlnames::PROP_ELEM);
  formater.write_fst_data(writer, fst);
  writer.end_element();
 
  for (int i = 0; i < size; ++i) {
    detail::xml_write_fst_state(writer, fst, i, formater);
  }
  writer.end_element();
}

template<typename Fst>
void xml_write_fst(xmlwriter & writer, const Fst & fst) {
  xml_write_fst(writer, fst, DefaultFstXmlFormater<Fst>());
}


template<typename Fst, typename XmlFormater>
void xml_write_fst(const boost::filesystem::path & p, const Fst & fst, XmlFormater formater) {
  xmlwriter writer(p);
  writer.set_indent(true);
  writer.start_document();
  xml_write_fst(writer, fst, formater);
  writer.end_document();
}

template<typename Fst>
void xml_write_fst(const boost::filesystem::path & p, const Fst & fst) {
  xml_write_fst(p, fst, DefaultFstXmlFormater<Fst>());
}



#endif
