#ifndef _BASIC_DATA_FST_H_
#define _BASIC_DATA_FST_H_

#include <vector>
#include <set>
#include <outilex/void_type.h>

#include <outilex/fst_output_ops.h>


/*
 * Generic FSA template implementation
 * * DataType is used to associate meta data to the states
 *   of some specialiazed instanciations (default to void_type)
 * * InputType is the input alphabet type
 */

template<typename InputType, typename OutputType,
         typename DataType = void_type>
class basic_fst {

public: // everything is public

  typedef basic_fst<InputType, OutputType, DataType> self;

  typedef DataType fst_data_type;

  typedef InputType  input_type;
  typedef OutputType output_type;

  struct label_type {
    input_type in;
    output_type out;

    label_type() : in(), out() {}
    label_type(const input_type & in_, const output_type & out_)
      : in(in_), out(out_) {}

    bool operator==(const label_type & b) const {
      return in == b.in && out == b.out;
    }
    bool operator!=(const label_type & b) const { return ! (*this == b); }

    bool operator<(const label_type & b) const {
      return (in < b.in) || (!(b.in < in) && (out < b.out));
    }
  };


  typedef std::set<output_type> outputs_type;

  struct data_type {

    outputs_type   final_outputs_;
    fst_data_type  data_; // meta-data
  
    data_type() : final_outputs_(), data_() {}

    template<typename ValueType>
    data_type(const ValueType & v)
      : final_outputs_(v.final_outputs().begin(), v.final_outputs().end()), data_(v.data()) {}

    outputs_type & final_outputs() { return final_outputs_; }
    const outputs_type & final_outputs() const { return final_outputs_; }

    fst_data_type & fst_data() { return data_; }
    const fst_data_type & fst_data() const { return data_; }

    /* assign operator */
    template<typename ValueType>
    data_type & operator=(const ValueType & v) {
      final_outputs_.clear();
      final_outputs_.insert(v.final_outputs.begin(), v.final_outputs.end());
      data_ = v.fst_data();
      return *this;
    }

    /* merge operator */
    template<typename ValueType>
    data_type & operator|=(const ValueType & v) {
      final_outputs_.insert(v.final_outputs().begin(), v.final_outputs().end());
      data_ |= v.fst_data();
      return *this;
    }
  };


  struct transition {

    label_type label_;
    int to_;

    transition() : label_(), to_(-1) {}

    transition(const label_type & label, int to)
      : label_(label), to_(to) {}
  
    template<typename Transition>
    transition(const Transition & tr) : label_(tr.in(), tr.out()), to_(tr.to()) {}
  
    transition(const input_type & in, const output_type & out, int to) : label_(in, out), to_(to) {}
 
    
    int to() const { return to_; }
    const label_type & label() const { return label_; }
    const input_type & in() const { return label_.in; }
    const output_type & out() const { return label_.out; }

    int & to() { return to_; }
    label_type & label() { return label_; }
    input_type & in() { return label_.in; }
    output_type & out() { return label_.out; }
  };


  typedef std::vector<transition> transitions;
  typedef typename transitions::iterator   trans_iterator;
  typedef typename transitions::const_iterator const_trans_iterator;


  static const int FLAG_FINAL = 1;

  struct state {

    int flags;
    data_type   data; // final-outputs + meta data ...
    transitions trans;

    state() : flags(0), data(), trans() {}
  };


  std::vector<state> states;

public:  /* constructors */

  basic_fst() : states() {}
  basic_fst(int size) : states(size) {}


public: /* modifiers */

  void clear() { states.clear(); }

  void reserve(int size) { states.reserve(size); }
  void resize(int size)  { states.resize(size);  }

  void trans_reserve(int q, int size) { states[q].trans.reserve(size); }
  void trans_resize(int q, int size)  { states[q].trans.resize(size);  }


  int add_state() {
    int res = size();
    resize(res + 1);
    return res;
  }



  /* flags */

  void set_final(int q, bool final = true) { 
    if (final) {
      states[q].flags |= FLAG_FINAL;
    } else { states[q].flags &= ~FLAG_FINAL; }
  }

  int & flags(int q) { return states[q].flags; }

  /* meta data setter access */

  /* data contains final outputs + meta-data
   * this useful to make fst instances work with generic fsa algorithms
   */

  data_type & data(int q) { return states[q].data; }

  template<typename ValueType>
  void set_data(int q, const ValueType & data) { states[q].data = data; }
 
  template<typename ValueType>
  void merge_data(int q, const ValueType & data) { states[q].data |= data; }

  /* final outputs */
  outputs_type & final_outputs(int q) { return states[q].data.final_outputs_; }

  /* data contains only meta data specified in DataType template parameter */

  fst_data_type & fst_data(int q) { return states[q].data.fst_data(); }

  template<typename ValueType>
  void set_fst_data(int q, const ValueType & data) { states[q].data.fst_data() = data; }

  template<typename ValueType>
  void merge_fst_data(int q, const ValueType & data) { states[q].data.fst_data() |= data; }


  /* transitions */
  trans_iterator trans_begin(int q) { return states[q].trans.begin(); }
  trans_iterator trans_end(int q) { return states[q].trans.end(); }

  transition & get_trans(int q, int transno) { return states[q].trans[transno]; }

  transitions & trans(int q) { return states[q].trans; }


  void add_trans(int from, const label_type & label, int to) {
    states[from].trans.push_back(transition(label, to));
  }

  void add_trans(int from, const input_type & in, const output_type & out, int to) { 
    states[from].trans.push_back(transition(in, out, to));
  }

  void swap(basic_fst & fst) { states.swap(fst.states); }

public:  /* queries */

  bool empty() const { return states.empty(); }
  int size() const { return states.size(); }
  int trans_size(int q) const { return states[q].trans.size(); }

  /* flags */
  int  flags(int q) const { return states[q].flags; }
  bool final(int q) const { return flags(q) & FLAG_FINAL; }

  /* meta data */
  const data_type & data(int q) const { return states[q].data; }
  const fst_data_type & fst_data(int q) const { return states[q].data.fst_data(); }


  /* final outputs */
  const outputs_type & final_outputs(int q) const { return states[q].data.final_outputs_; }

  typename outputs_type::const_iterator final_outputs_begin(int q) const {
    return states[q].data.final_outputs_.begin();
  }
  typename outputs_type::const_iterator final_outputs_end(int q) const {
    return states[q].data.final_outputs_.end();
  }

  /* transitions iteration */
  const_trans_iterator trans_begin(int q) const { return states[q].trans.begin(); }
  const_trans_iterator trans_end(int q) const { return states[q].trans.end(); }

  const transition & get_trans(int q, int transno) const { return states[q].trans[transno]; }
};


#endif

