#ifndef _BASIC_DATA_FSA_H_
#define _BASIC_DATA_FSA_H_

#include <vector>
#include <outilex/void_type.h>

/*
 * Generic FSA template implementation
 * * DataType is used to associate meta data to the states
 *   of some specialiazed instanciations (default to void_type)
 * * InputType is the input alphabet type
 */

template<typename InputType,
         typename DataType = void_type>
class basic_fsa {

public: // everything is public

  typedef basic_fsa<InputType, DataType> self;

  typedef DataType data_type;

  typedef InputType input_type;
  typedef InputType label_type;

  struct transition {

    input_type label_;
    int to_;

    transition() : label_(), to_() {}
    transition(const input_type & label, int to) : label_(label), to_(to) {}
  
  
    int to() const { return to_; }
    const label_type & label() const { return label_; }
    const input_type & in() const { return label_; }

    int & to() { return to_; }
    label_type & label() { return label_; }
    input_type & in() { return label_; }
  };


  typedef std::vector<transition> transitions;
  typedef typename transitions::iterator   trans_iterator;
  typedef typename transitions::const_iterator const_trans_iterator;


  static const int FLAG_FINAL = 1;

  struct state {

    int flags;
    data_type   data;
    transitions trans;

    state() : flags(0), data(), trans() {}
  };


  std::vector<state> states;

public:  /* constructors */

  basic_fsa() : states() {}
  basic_fsa(int size) : states(size) {}


public: /* modifiers */

  void clear() { states.clear(); }

  void reserve(int size) { states.reserve(size); }
  void resize(int size)  { states.resize(size);  }

  void trans_reserve(int q, int size) { states[q].trans.reserve(size); }
  void trans_resize(int q, int size)  { states[q].trans.resize(size);  }


  int add_state() {
    int res = size();
    resize(res + 1);
    return res;
  }



  /* flags */

  void set_final(int q, bool final = true) { 
    if (final) {
      states[q].flags |= FLAG_FINAL;
    } else { states[q].flags &= ~FLAG_FINAL; }
  }

  int & flags(int q) { return states[q].flags; }

  /* meta data access */

  data_type & data(int q) { return states[q].data; }

  template<typename ValueType>
  void set_data(int q, const ValueType & data) { states[q].data = data; }

  template<typename ValueType>
  void merge_data(int q, const ValueType & data) { states[q].data |= data; }


  /* transitions */
  trans_iterator trans_begin(int q) { return states[q].trans.begin(); }
  trans_iterator trans_end(int q) { return states[q].trans.end(); }

  transition & get_trans(int q, int transno) { return states[q].trans[transno]; }

  void add_trans(int from, const label_type & label, int to) {
    states[from].trans.push_back(transition(label, to));
  }

  void swap(self & fsa) { states.swap(fsa.states); }

public:  /* queries */

  bool empty() const { return states.empty(); }
  int size() const { return states.size(); }
  int trans_size(int q) const { return states[q].trans.size(); }

  /* flags */
  int  flags(int q) const { return states[q].flags; }
  bool final(int q) const { return flags(q) & FLAG_FINAL; }

  /* meta data */
  const data_type & data(int q) const { return states[q].data; }

  /* transitions iteration */
  const_trans_iterator trans_begin(int q) const { return states[q].trans.begin(); }
  const_trans_iterator trans_end(int q) const { return states[q].trans.end(); }

  const transition & get_trans(int q, int transno) const { return states[q].trans[transno]; }
};

#endif
