#include <iostream>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>

#include <outilex/generic_fst.h>
#include <outilex/fsa-prune.h>
#include <outilex/epsilon_removal.h>
#include <outilex/fsa-determinize.h>

using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

namespace {

char * progname;

void usage() {
  cout << "usage : " << progname << " [-r|-p|-d] [-o <gfst> ] <gfst>\n";
  exit(1);
}

bool is_epsilon(const generic_fst::transition & tr) {
  return (tr.in() == "<E>" || tr.in().empty()) && tr.out().empty();
}

} // namespace ""

int main(int argc, char ** argv) try {

  fs::path ipath, opath;
  bool epsilon_cleanup = false;
  bool prune_cleanup = false;
  bool determinize = false;

  progname = *argv;
  argv++, argc--;

  if (! argc) { usage(); }

  while (argc) {
  
    string arg = *argv;

    if (arg == "-h") {
    
      usage();
    
    } else if (arg == "-p") {

      prune_cleanup = true;

    } else if (arg == "-r") {

      epsilon_cleanup = true;

    } else if (arg == "-d") {

      determinize = true;

    } else if (arg == "-o") {
    
      argv++, argc--;
      if (! argc) { cerr << "bad args\n"; exit(1); }

      opath = fs::path(*argv, fs::native);
    
    } else {
      ipath = fs::path(*argv, fs::native);
    }

    argv++, argc--;
  }

  if (ipath.empty()) {
    cerr << "arg missing\n";
    exit(1);
  }

  if (opath.empty()) { opath = ipath; }

  cerr << "loading " << ipath.string() << endl;

  generic_fst fst(ipath);

  if (prune_cleanup) {
    cerr << "pruning ...\n";
    fsa_prune(fst);
  }

  if (epsilon_cleanup) {
    cerr << "epsilon_removal ...\n";
    fsa_remove_epsilon(fst, is_epsilon);
    cerr << "(re-)pruning...\n";
    fsa_prune(fst);
  }

  if (determinize) {
    cerr << "determinization...\n";
    fsa_determinize(fst);
  }
  cerr << "done.\n";

  fst.write(opath);

} catch (exception & e) {
  cerr << "fatal error: " << e.what() << endl;
  exit(1);
}

