#include <map>
#include <vector>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/lexical_cast.hpp>

#include <outilex/generic_fst.h>
#include <outilex/unitex_grf.h>
#include <outilex/flatten.h>
#include <outilex/fsa-prune.h>
#include <outilex/epsilon_removal.h>
#include <outilex/fsa-determinize.h>
#include <outilex/fsa-minimize.h>
#include <outilex/fsa_nb_transitions.h>

#include <outilex/usage.h>


using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

const char * USAGE_DESCRIPTION =
"usage: flatten [-v] <grf8>\n"
"\n"
"with\n"
" -v  : verbose mode\n"
" -max <n> : stop at depth <n>\n"
"\n"
"Create a generic fst from a grf8 file.\n"
"The subgraphs are flattening into the main fst.\n"
"\n";


namespace {

bool is_epsilon(const generic_fst::transition & tr) {
  return (tr.in() == "<E>" || tr.in().empty()) && tr.out().empty();
}

void count_final(generic_fst & fst) {
  int nb = 0;
  for (int q = 0; q < fst.size(); ++q) {
    if (fst.final(q)) { nb++; }
  }
  cout << nb << " finals\n";
}

} // anonymous namespace



int main(int argc, char ** argv) try {

  fs::path::default_name_check(fs::no_check);

  fs::path fstpath;
  bool verbose = false;
  int maxdepth = std::numeric_limits<int>::max();

  argv++, argc--;
  
  if (argc == 0) { usage(); }

  while (argc) {
    string arg = *argv;
    
    if (arg[0] == '-') {
    
      if (arg == "-v") {
        verbose = true;
      } else if (arg == "-max") {
        argv++,argc--;
        if (argc == 0) { arg_needed(arg); }
        maxdepth = lexical_cast<int>(*argv);

      } else if (arg == "-h") {
        usage();
      } else {
        bad_args("unknown argument '" + arg + "'");
      }
    } else { fstpath = arg; }
 
    argv++, argc--;
  }
  
  if (fstpath.empty()) {
    bad_args();
  }

  generic_fst fst;
  string fname  = fstpath.leaf();

  string::size_type dot = fname.rfind('.');

  if (verbose) { cout << "loading " << fstpath.string() << "\n"; }

  if (dot != string::npos && fname.substr(dot) == ".grf8") {
  
    unitex_grf grf(fstpath);
    grf_to_generic_fst(grf, fst);

  } else {
    fst.read(fstpath);
  }
  if (verbose) { cout << "done. (" << fst.size() << " states).\n"; }


  if (verbose) { cout << "flattening ...\n"; }
  bool ok = flatten(fst, fstpath.branch_path(), maxdepth);
  if (verbose) { 
    cout << "done. (" << fst.size() << " states, " << fsa_nb_trans(fst) << " trans).\n";
    if (ok) {
      cout << "result is an equivalent FST.\n";
    } else {
      cout << "result is an approximation.\n";
    }
  }

  //cout << "cleaning ...\n";

  if (verbose) { cout << "pruning ...\n"; }
  fsa_prune(fst);
  if (verbose) { 
    cout << "done. (" << fst.size() << " states, " << fsa_nb_trans(fst) << " trans).\n";
  }

  if (verbose) { cout << "epsilon removal ...\n"; }
  fsa_remove_epsilon(fst, is_epsilon);
  if (verbose) { 
    cout << "done. (" << fst.size() << " states, " << fsa_nb_trans(fst) << " trans).\n";
  }

  if (verbose) { cout << "determinization ...\n"; }
  fsa_determinize(fst);
  if (verbose) { 
    cout << "done. (" << fst.size() << " states, " << fsa_nb_trans(fst) << " trans).\n";
  }

  if (verbose) { cout << "minimization ...\n"; }
  fsa_minimize(fst);
  if (verbose) { 
    cout << "done. (" << fst.size() << " states, " << fsa_nb_trans(fst) << " trans).\n";
  }


  fstpath = change_extension(fstpath, ".gfst");
  fst.write(fstpath);

  cout << "done. result in " << fstpath.string() << " (" << fst.size() << " states, "
    << fsa_nb_trans(fst) << " transitions)\n";

} catch (exception & e) {
  cerr << "fatal error: " << e.what() << endl;
  exit(1);
}

