#include <iostream>


#include <outilex/sentence_fsa.h>
#include <outilex/wrtn_chart.h>
#include <outilex/wrtn_parser.h>
#include <outilex/wrtn_transduct.h>


using namespace std;

namespace {


void output_match_tree(const sentence_fsa & text, const wrtn_match & match,
                       const wrtn_chart & chart, ostream & os) {

  os << "(" << match.name << ' ';

  int size = match.path.size();

  for (int i = 0; i < size; ++i) {
    const syntref & ref = match.path[i];
    if (ref.transno < 0) {
      output_match_tree(text, chart[ref.qno][- ref.transno - 1], chart, os);
      os << ' ';
    } else {
      //os << 'x' << text.get_trans(ref.qno, ref.transno).in() << 'x' << ' ';
      os << text.get_trans(ref.qno, ref.transno).in().form << ' ';
    }
  }

  os << " " << match.name << ")";
}

void dump_all_matches(const sentence_fsa & text, const wrtn_chart & chart,
                      const string & syntname, ostream & os) {

  os << "dump all matches :\n";

  for (int q = 0; q < text.size(); ++q) {
    os << q << ":\n";
    for (wrtn_chart::const_match_iterator it = chart[q].begin();
         it != chart[q].end(); ++it) {
      if (it->name == syntname) {
        output_match_tree(text, *it, chart, os);
        os << '\n';
      }
    }
  }
}


void output_match(const sentence_fsa & text, const wrtn_match & match,
                  const wrtn_chart & chart, ostream & os,
                  trans_mode_type transmode, trans_output_type outmode) {

  if (outmode == HTML) { os << "<strong>"; }


  //output_match_tree(text, match, chart, os);

  int size = match.path.size();
  int i = 0;
  for (i = 0; i < size; ++i) {
    
    const syntref & ref = match.path[i];
    
    if (! match.out[i].empty()) { os << match.out[i] << ' '; }
 
    if (ref.transno < 0) {
      output_match(text, chart[ref.qno][- ref.transno - 1], chart, os, transmode, outmode);
    } else {
      os << text.get_trans(ref.qno, ref.transno).in().form << ' ';
    }
  }

  if (! match.out[i].empty()) { os << match.out[i] << ' '; }
  //os << "/" << match.w;
  if (outmode == HTML) { os << "</strong>"; }
}

void output_trans(const sentence_fsa::transition & tr, ostream & os,
                  trans_mode_type transmode, trans_output_type outmode) {
  os << tr.in().form << ' ';
}


struct compare_dest {
  template<typename Trans>
  bool operator()(const Trans & a, const Trans & b) {
    return a.to() < b.to();
  }
};


wrtn_chart::const_match_iterator find_better_match(const string & syntname, 
                                                   const unique_sequence<wrtn_match> & tab) {

  wrtn_chart::const_match_iterator res = tab.end();
  int to = -1;
  double w = 0.0;

  for (wrtn_chart::const_match_iterator it = tab.begin(), end = tab.end();
       it != end; ++it) {
    
    if (it->name == syntname && it->to >= to) {
    
      if (it->to > to) { // longest match first
          to = it->to; w = it->w;
          res = it;
      } else { // it->to == to, best match == higher weight
        if (it->w > w) {
          to = it->to; w = it->w;
          res = it;
        }
      }
    }
  }
  //cout << "(" <<w<<")"<<endl;
  return res;
}


}; // namespace ""

int wrtn_transduct(sentence_fsa & text_, wrtn_parser & parser,
                  ostream & os, trans_mode_type transmode, trans_output_type outmode) {

  //  cerr << "\n\ntransduct new sentence\n";
  //  cerr << text.text << "\n\n";

  if (text_.empty()) {
    cerr << "warning: empty sentence\n";
    return 0;
  }

  wrtn_chart chart;
  chart.eat(text_);

  const sentence_fsa & text = chart.fsa;
  parser.parse(chart, true);
  
  int nbmatches = 0;
  const string & syntname = parser.gram.start_name();

  //dump_all_matches(text, chart, syntname, cerr);

  int q = 0;
  while (! text.final(q)) {
    
    wrtn_chart::const_match_iterator m = find_better_match(syntname, chart[q]);
    if (m != chart[q].end()) { // find a match
      
      ++nbmatches;
      
      output_match(text, *m, chart, os, transmode, outmode);
      q = m->to;

    } else { // output a transition

#warning : todo output lex transition

      sentence_fsa::const_trans_iterator tr = min_element(text.trans_begin(q),
                                                          text.trans_end(q),
                                                          compare_dest());
    
      if (tr == text.trans_end(q)) {
        os << "*error*";
        cerr << "error: bad text automaton\n";
        goto out;
      }
      output_trans(*tr, os, transmode, outmode);
      q = tr->to();
    }
  }

out:
  os << "\n";
  if (outmode == HTML) {
    os << "<br>\n";
  }
  return nbmatches; 
}

