#include <iostream>
#include <fstream>
#include <stdexcept>
#include <sstream>
#include <vector>

#include <boost/progress.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/fstream.hpp>

#include <outilex/text_fsa.h>
#include <outilex/sentence_fsa.h>
#include <outilex/wrtn_grammar.h>
#include <outilex/wrtn_chart.h>
#include <outilex/wrtn_parser.h>
#include <outilex/wparsing_helper.h>
#include <outilex/text_transduction.h>



using namespace std;
using namespace boost;
namespace fs = boost::filesystem;


namespace {

char * progname;


void usage() {
  cerr << "usage: " << progname
    << " -l <lingdef> -gram <fst> [-ipath|-iout][-html|-txt][-m|-r|-i][-o <outputres>] <txtfsa>\n";
  exit(1);
}

}; // namespace ""


int main(int argc, char ** argv) try {

  fs::path txtpath, lingdefpath, grampath, opath;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }

  txt_trans_mode_type transmode    = MERGE;
  txt_trans_output_type outputmode = TEXT;

  int PARSER_FLAGS = wrtn_parser::SURF_PARSING;

  progname = *argv;

  argv++, argc--;

  while (argc) {
    
    string arg = *argv;
    
    if (arg == "-l") {
      
      argv++, argc--;
      if (argc == 0) { usage(); }
      lingdefpath = fs::path(*argv, fs::native);
 
    } else if (arg == "-o") {
      
      argv++, argc--;
      if (argc == 0) { usage(); }
      opath = fs::path(*argv, fs::native);
    
    } else if (arg == "-gram") {
 
      argv++, argc--;
      if (argc == 0) { usage(); }
      grampath = fs::path(*argv, fs::native);
    
    } else if (arg == "-html") {
    
      outputmode = HTML;
    
    } else if (arg == "-text") {
    
      outputmode = TEXT;
    
    } else if (arg == "-m") {
    
      transmode = MERGE;
    
    } else if (arg == "-r") {
    
      transmode = REPLACE;
    
    } else if (arg == "-i") {
    
      transmode = IGNORE;

    } else if (arg == "-ipath") {
    
      PARSER_FLAGS |= wrtn_parser::IGNORE_DIFF_PATHS;

    } else if (arg == "-iout") {
    
      PARSER_FLAGS |= wrtn_parser::IGNORE_DIFF_OUTPUTS;

    } else if (arg == "-h") {
    
      usage();
    
    } else {

      txtpath = fs::path(arg, fs::native);
    }

    argv++, argc--;
  }

  if (txtpath.empty() || lingdefpath.empty() || grampath.empty()) { usage(); }

  if (opath.empty()) {

    string resname = txtpath.leaf();

    resname += ".trans";

    if (outputmode == HTML) {
      resname += ".html";
    } else {
      resname += ".txt";
    }
    opath = txtpath.branch_path() / resname;
  }
 
  fs::ofstream os(opath);
  if (! os) { cerr << "unable to open " << opath.string() << endl; exit(1); }

  if (outputmode == HTML) {
    os << 
      "<html>\n"
      "<head>\n"
      "<meta http-equiv=\"Content-Type\" content=\"text/html;charset=utf-8\"/>\n"
      "<title>" << txtpath.leaf() << " : Transduction by "
      << grampath.leaf() << "</title>\n" <<
      "</head>\n"
      "<body bgcolor='#ffffff' text='#000000'>\n"
      "<h1>" << txtpath.leaf() << " transducted</h1>\n";
  }


  ling_def lingdef(lingdefpath);

  scoped_ptr<itext_fsa> p_itext(new_itext_fsa(txtpath, & lingdef));
  itext_fsa & itext = *p_itext;

  wrtn_grammar gram(grampath, & lingdef);

  int total = itext.size();

  wchart_text_transducer transducer(os, gram.start_name(), transmode, outputmode);
  progress_displayer<wchart_text_transducer> displayer(transducer, total);

  boost::timer tmr;
  wrtn_parse(itext, gram, displayer, PARSER_FLAGS, true); // true for longest_match


  if (outputmode == HTML) {
    os << "<br/>\n" << transducer.nbmatch << " matching sequences\n";
    os << "</body>\n</html>\n";
  }


  cout << "done. " << displayer.count() << "/" << total << " sentences parsed.\n"
    << transducer.nbmatch << " matching sequences. " << tmr.elapsed() << "s.\n"
    << "result in " << opath.string() << "\n";


} catch (exception & e) {

  cerr << "exception caught :" << e.what() << endl; exit(1);

} catch (...) { cerr << "caught an OVNI?\n"; exit(1); }

