#include <iostream>
#include <string>

#include <outilex/wrtn_chart.h>
#include <outilex/text_transduction.h>

using namespace std;

namespace {


struct same_pos {

  pos_def * pos;

  same_pos(pos_def * p) : pos(p) {}

  template<typename Trans>
  bool operator()(const Trans & tr) const { return tr.in().pos == pos; }
};

struct compare_dest {
  template<typename Trans>
  bool operator()(const Trans & a, const Trans & b) const { return a.to() < b.to(); }
};


sentence_fsa::const_trans_iterator find_best_trans(const sentence_fsa & fsa, int q) {

  const sentence_fsa::const_trans_iterator begin = fsa.trans_begin(q), end = fsa.trans_end(q);

  /* first, try to find a lex transition */
  pos_def * lexpos = fsa.lingdef->lex_pos();

  sentence_fsa::const_trans_iterator tr = find_if(begin, end, same_pos(lexpos));
  
  if (tr != end) { // found
    return tr;
  }
  
  /* else, return the shortest one */

  return min_element(begin, end, compare_dest());
}


inline void output_trans(const sentence_fsa::transition & tr, ostream & os) {
  os << tr.in().form << ' ';
}

void output_match(const wrtn_chart & chart, const wrtn_match & match, ostream & os,
                  txt_trans_mode_type transmode) {

  int size = match.path.size();
  for (int i = 0; i < size; ++i) {
    
    const syntref & ref = match.path[i];
    
    if (transmode != IGNORE && ! match.out[i].empty()) { os << match.out[i] << ' '; }
 
    if (ref.transno < 0) {
      output_match(chart, chart[ref.qno][- ref.transno - 1], os, transmode);
    } else if (transmode != REPLACE) {
      output_trans(chart.fsa.get_trans(ref.qno, ref.transno), os);
    }
  }

  if (transmode != IGNORE && ! match.out[size].empty()) { os << match.out[size] << ' '; }
}


} // namespace ""

int wchart_text_transduct(const wrtn_chart & chart, ostream & os,
                          const string & axiom,
                          txt_trans_mode_type transmode, txt_trans_output_type outmode) {

  const sentence_fsa & text = chart.fsa;

  if (text.empty()) { return 0; }
  

  /* wrtn_match directly inferior (<) to the first syntname match in chart. */
  const wrtn_match firstmatch(axiom,
                              numeric_limits<int>::max(),
                              synt_path_type(),
                              vector<string>(),
                              numeric_limits<double>::max());

  int nbmatch = 0;

  int q = 0;
  while (! text.final(q)) {

    /* search for a best match */
    wrtn_chart::match_by_val_iterator it = chart[q].lower_bound(firstmatch);

    if (it != chart[q].val_end() && it->name == axiom) { // we find it
 
      if (outmode == HTML) { os << "<strong>"; }

      output_match(chart, *it, os, transmode);

      if (outmode == HTML) { os << "</strong>"; }

      ++nbmatch;
      q = it->to;

    } else { // output a transition
    
      sentence_fsa::const_trans_iterator tr = find_best_trans(text, q); 
    
      if (tr == text.trans_end(q)) {
        cerr << "error: in text_transduct: mal-formed text automaton\n";
        os << "*ERROR*";
        goto out;
      }

      output_trans(*tr, os);
      q = tr->to();
    }
  }

out:
  os << "\n";
  if (outmode == HTML) { os << "<br>\n"; }
  return nbmatch;
}

