#include <iostream>
#include <string>
#include <vector>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>

#include <outilex/lingdef.h>
#include <outilex/wrtn_grammar.h>
#include <outilex/unitex_grf.h>
#include <outilex/generic_fst.h>
#include <outilex/fsa-prune.h>
#include <outilex/fsa-determinize.h>
#include <outilex/epsilon_removal.h>


using namespace std;
using namespace boost;

namespace fs = boost::filesystem;

namespace {

template<typename OutputIterator>
int lookup_subpatterns(const wrtn_pattern & fst, OutputIterator out) {

  int nbtrans = 0;
  for (int q = 0; q < fst.size(); ++q) {
    
    for (wrtn_pattern::transitions::const_iterator tr = fst.trans_begin(q);
         tr != fst.trans_end(q); ++tr) {
      
      if (tr->in().type == wrtn_input_type::SUBCALL) {
        const string & name = boost::get<string>(tr->in().v);
        //cout << name << " " << flush;
        *out = name;
        ++out;
      }
      ++nbtrans;
    }
  }
  return nbtrans;
}

void cleanup_pattern(wrtn_pattern & pat, vector<string> & badnames) {

  for (int q = 0; q < pat.size(); ++q) {

    for (wrtn_pattern::transitions::iterator tr = pat.trans_begin(q);
         tr != pat.trans_end(q);) {
      
      if (tr->in().type == wrtn_input_type::SUBCALL
          && find(badnames.begin(), badnames.end(), boost::get<string>(tr->in().v))
             != badnames.end()) {

        tr = pat.trans(q).erase(tr);
      } else {
        ++tr;
      }
    }
  }
}

void cleanup_grammar(wrtn_grammar & gram, vector<string> & badnames) {

  for (int i = 0; i < gram.size(); ++i) {
    cleanup_pattern(gram[i], badnames);
  }
}


char * progname;

void usage() {
  cout << "usage: " << progname << " [-l <lingdef>][-f] <axiom>\n";
  exit(0);
}


time_t lingdef_time;
bool force_remake = false;

/* remake wpat file, if xgrf or grf8 is newer
 */


bool is_epsilon(const generic_fst::transition & tr) {
  return (tr.in() == "<E>" || tr.in().empty()) && tr.out().empty();
}


void update_wpat(const fs::path & fstpath, const fs::path & grfpath, const fs::path & xgrfpath,
                 ling_def * lingdef) {

  time_t last_edit = 0;
  fs::path path;

  if (fs::exists(grfpath)) {
    path = grfpath;
    last_edit = fs::last_write_time(grfpath);
  }

  if (fs::exists(xgrfpath)) { 

    time_t xgrf = fs::last_write_time(xgrfpath);
 
    if (path.empty() || xgrf > last_edit) {
      path = xgrfpath;
      last_edit = xgrf;
    }
  }

  if (path.empty()) { // none exists
    return;
  }


  if (force_remake
      || ! fs::exists(fstpath)
      || fs::last_write_time(fstpath) <= last_edit
      || fs::last_write_time(fstpath) <= lingdef_time) { // (re)make wpat
  
    cout << "updating from " << path.string() << "... ";

    unitex_grf grf(path);

    generic_fst gfst;
    grf_to_generic_fst(grf, gfst);

    // fst cleanup

    fsa_prune(gfst);
    fsa_remove_epsilon(gfst, is_epsilon);
    fsa_prune(gfst);
    fsa_determinize(gfst);
  
    gfst.name = fs::basename(fstpath);

    wrtn_pattern(gfst, lingdef).write(fstpath);
  }
}
} // namespace ""


int main(int argc, char ** argv) try {

  fs::path fstpath, lingdefpath;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }

  progname = *argv;
  argv++, argc--;

  if (argc == 0) { usage(); }

  while (argc) {
  
    string arg = *argv;

    if (arg == "-l") {

      argv++, argc--;
      if (argc == 0) { usage(); }
      lingdefpath = fs::path(*argv, fs::native);

    } else if (arg == "-h") {
    
      usage();
    
    } else if (arg == "-f") {

      force_remake = true;

    } else {
      fstpath = fs::path(arg, fs::native);
    }
    argv++, argc--;
  }

  if (fstpath.empty() || lingdefpath.empty()) { cerr << "bad args\n"; exit(1); }

  ling_def * lingdef = new ling_def(lingdefpath); 

  // time of the last modif of th lingdef
  lingdef_time = fs::last_write_time(lingdefpath);

  wrtn_grammar gram(lingdef);

  fs::path grampath = fs::change_extension(fstpath, ".wrtn");


  fs::path dir(fstpath.branch_path());

  string maingraph = fs::basename(fstpath);

  vector<string> stack;
  vector<string> badnames;

  stack.push_back(maingraph);

  int totalstates = 0, totaltrans = 0;

  while (! stack.empty()) {

    string name = stack.back();
    stack.pop_back();
  
    if (gram.get_pat_idx(name) != -1) { continue; } // already proceeded

    cout << "processing " << name << " ... " << flush;
  
    fs::path fstpath = dir / (name + ".wpat");
    fs::path grfpath = fs::change_extension(fstpath, ".grf8");
    fs::path xgrfpath = fs::change_extension(fstpath, ".xgrf");
  
    update_wpat(fstpath, grfpath, xgrfpath, lingdef);

    if (! fs::exists(fstpath)) {
      cerr << "error: unable to find " << fstpath.string() << endl;
      badnames.push_back(name);
      continue;
    }

    int idx = gram.add_pattern(fstpath, name);
    int nbtrans = lookup_subpatterns(gram[idx], back_inserter(stack));

    cout << '(' << gram[idx].size() << " states, " << nbtrans << " transitions)\n";

    totalstates += gram[idx].size();
    totaltrans  += nbtrans;
  }

  if (! badnames.empty()) {
    cout << "grammar cleanup ...\n";
    cleanup_grammar(gram, badnames);
  }
  cout << "writing grammar into " << grampath.string() << '\n'
    << gram.size() << " patterns\n"
    << "total of " << totalstates << " states and " << totaltrans << " transitions.\n";

  gram.write(grampath);
  cout << "done.\n";

  return 0;

} catch (exception & e) {
  cerr << "fatal error: " << e.what() << endl;
  exit(1);
}

