#include <iostream>

#include <outilex/wrtn_chart.h>
#include <outilex/fsa_decoration.h>

#include <outilex/stringtok.h>
#include <outilex/escape.h>


using namespace std;

namespace {


void output_match(const wrtn_chart & chart, const wrtn_match & match, ostream & os) {

  int size = match.path.size();
  for (int i = 0; i < size; ++i) {
    
    const syntref & ref = match.path[i];
    
    if (! match.out[i].empty()) { os << "'" << match.out[i] << "'"; }
 
    if (ref.transno < 0) {
      output_match(chart, chart[ref.qno][- ref.transno - 1], os);
    } else {
      os << chart.fsa.get_trans(ref.qno, ref.transno).in().form << ' ';
    }
  }

  if (! match.out[size].empty()) { os << "'" << match.out[size] << "'"; }
}



void add_feats_from_output(ostringstream & code, string & lemma,
                           const sentence_fsa & text,
                           const syntref & ref, const string & label) {

  //cerr << "add_feats: code = " << code.str() << " label = '" << label << "'\n";

  vector<string> codes;
  stringtok(label, "+,[]", back_inserter(codes));

  for (int n = 0; n < codes.size(); ++n) {

    if (codes[n].empty()) { continue; }

    if (codes[n][0] == '^') { // retrieve feat from the lexical entry

      if (ref.transno < 0) {
        cerr << "error: in label " << label << ": " << codes[n] << " under a subcall\n";
        continue;
      }

      const lexical_mask & m = text.get_trans(ref.qno, ref.transno).in();
      const string attrname = codes[n].substr(1);

      if (attrname == "lemma") {

        lemma = m.lemma;

      } else {

        const feat_set & fs = m[attrname];
        code << "+" << attrname << "=" << fs.get_val_text();
      }

    } else {
      code << "+" << codes[n];
    }
  }
  //cerr << "out of add_feats: code = " << code.str() << " label = '" << label << "'\n\n";
}


int process_match(wrtn_chart & chart, const wrtn_match & match,
                  int q0, string & form, string & lemma, ostringstream & code,
                  mutable_lexic & lexic, bool verbose) {

  sentence_fsa & text = chart.fsa;
  ling_def * ldef = text.lingdef;

  int size = match.path.size();
  int nbmatch = 0;

  int i = 0; // we need it outside loop
  for (i = 0; i < size; ++i) {

    const syntref & ref = match.path[i];
    string out = match.out[i];

    //cerr << "out = " << out << "\n\n";

    string::size_type eot = out.find(']'); // eot stands for end of tag

    /* end of a tag */
    if (q0 >= 0 
        && eot != string::npos) {

      form.resize(form.size() - 1); // strip ending ' '

      add_feats_from_output(code, lemma, text, ref, out.substr(0, eot));

      if (lemma.empty()) { lemma = form; }

      string txt = "{" + form + "," + lemma + "." + code.str().substr(1) + "}";
      
      if (verbose) { cerr << "find something : " << txt << "\n"; }
 
      try {

        lexical_mask m(txt, ldef);
        if (! m) { throw runtime_error("invalid entry"); }

        if (verbose) { cerr << "lexmask = " << m << "\n"; }

        int lbl = lexic.add(m);
        text.A.add_trans(q0, lbl, match.path[i].qno); // add transition

        ++nbmatch;

      } catch (exception & e) {
        cerr << "error with " << txt << ": " << e.what() << endl;
      }

      q0 = -1;
      out = out.substr(eot + 1);
    }

    string::size_type bot = out.find('[');

    /* begin of a tag */
    if (q0 == -1
        && bot != string::npos) {

      form.clear(); code.str(""); lemma.clear();
      q0 = match.path[i].qno;
      out = out.substr(bot);
    }

 

    /* travel */

    if (ref.transno < 0) { // sub-pattern

      nbmatch += process_match(chart, chart[ref.qno][-ref.transno - 1],
                               (q0 == -1) ? -1 : -2, form, lemma, code, lexic, verbose);

      // -2 c'est pour forcer que les sortie '[' et ']' soient dans le meme graphe
 
    } else { // lexical trans

      if (q0 != -1) { // -2 ou >= 0
        form += escape(text.get_trans(ref.qno, ref.transno).in().form, ",.") + " ";
      }
    }

    if (q0 != -1 && ! out.empty()) {
      add_feats_from_output(code, lemma, text, ref, out);
    }
  }


  /* end of match : look for last output
   */

  const string & out = match.out[i];
  if (q0 >= 0
      && ! out.empty() && out[out.size() - 1] == ']') { // end of tag

    form.resize(form.size() - 1); // strip ending ' '

    if (out.size() > 1) {
      add_feats_from_output(code, lemma, text,
                            syntref(0, -1),     // fake syntref
                            out);
    }
 
    if (lemma.empty()) { lemma = form; }

    string txt = "{" + form + "," + lemma + "." + code.str().substr(1) + "}";
  
    if (verbose) { cerr << "find something (end) : " << txt << "\n"; }

    try {

      lexical_mask m(txt, ldef);
      if (! m) { throw runtime_error("invalid entry"); }

      if (verbose) { cerr << "lexmask = " << m << endl; }

      int lbl = lexic.add(m);
      text.A.add_trans(q0, lbl, match.to); // add transition

      ++nbmatch;

    } catch (exception & e) {
      cerr << "(at end of match) error with " << txt << ": " << e.what() << endl;
    }
    q0 = -1;
  }

  if (q0 == -2 && ! out.empty()) {
    add_feats_from_output(code, lemma, text, syntref(0,-1), out);
  }

  if (q0 >= 0) {
    cerr << "error: tag '[' and ']' mismatch :";
    output_match(chart, match, cerr);
    cerr << '\n';
  }

  return nbmatch;
}


int process_match(wrtn_chart & chart, const wrtn_match & match, mutable_lexic & lexic,
                  bool verbose) {

  string form, lemma;
  ostringstream code;
  return process_match(chart, match, -1, form, lemma, code, lexic, verbose);
}


}; // namespace ""



int decore_fsa(wrtn_chart & chart, const string & axiom, bool verbose) {

  sentence_fsa & fsa = chart.fsa;

  mutable_lexic lexic;
  fsa.strip_lexic(lexic);

  int nbmatch = 0;
  int size = chart.size();

  for (int q = 0; q < size; ++q) {
 
    for (wrtn_chart::const_match_iterator it = chart[q].begin(), end = chart[q].end();
         it != end; ++it) {
    
      if (it->name == axiom) { nbmatch += process_match(chart, *it, lexic, verbose); }
    }
  }

  fsa.set_lexic(lexic);
  fsa.determinize();
  fsa.topological_sort();

  return nbmatch;
}

