#include <iostream>
#include <fstream>
#include <stdexcept>
#include <sstream>
#include <vector>

#include <boost/progress.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/lexical_cast.hpp>

#include <outilex/text_fsa.h>
#include <outilex/bin_text_fsa.h>
#include <outilex/sentence_fsa.h>
#include <outilex/wrtn_grammar.h>
#include <outilex/wrtn_chart.h>
#include <outilex/escape.h>
#include <outilex/stringtok.h>
#include <outilex/wparsing_helper.h>

#include <outilex/usage.h>

using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

const char * USAGE_DESCRIPTION =
"usage: chunker -l <lingdef> -gram <wrtn> [-maxtime <s>][-v][-ipath][-iout][-o <res>] <txtfsa>\n";


namespace {


void output_match(const wrtn_match & match, const wrtn_chart & chart, ostream & os) {

  const sentence_fsa & text = chart.fsa;

  int size = match.path.size();

  for (int i = 0; i < size; ++i) {
    
    const syntref & ref = match.path[i];

    if (ref.transno < 0) {
      output_match(chart[ref.qno][- ref.transno - 1], chart, os);
    } else {
      os << text.get_trans(ref.qno, ref.transno).in() << ' ';
    }
  }
}




int output_matches(const wrtn_chart & chart,
                   const string & axiom, ostream & os) {
                   
  const sentence_fsa & text = chart.fsa;

  //map<string, int> matches_map;

  int nbmatch = 0;

  //find best matches

  int maxq = -1;
  double maxw = -1;
  for (wrtn_chart::const_match_iterator it = chart[0].begin(), end = chart[0].end();
       it != end; ++it) {

    if (it->name != axiom) { continue; }

    if (it->to > maxq) {
      maxq = it->to;
      maxw = it->w;
    } else if (it->to == maxq && it->w > maxw) { maxw = it->w; }
  }

  for (wrtn_chart::const_match_iterator it = chart[0].begin(), end = chart[0].end();
       it != end; ++it) {

    if (it->name == axiom && it->to == maxq && it->w == maxw) {    

      //ostringstream oss;
      output_match(*it, chart, os);

      //matches_map[oss.str()]++;

      if (! text.final(it->to)) {
        os << " ...\n\n";
      } else {
        os << "\n\n";
        ++nbmatch;
      }
    }
  }

#if 0
  for (map<string,int>::iterator it = matches_map.begin(), end = matches_map.end();
       it != end; ++it) {
  
    os << it->first << (text.final(maxq) ? " " : " ... ");
    /*if (it->second > 1)*/ { os << "(x" << it->second << ")"; }
    os << "\n\n";
    ++nbmatch;
  }
#endif
  return nbmatch;
}

struct wchart_chunker {

  ostream & out;
  ostream & errors;
  const string & axiom;
  bool verbose;
  int sentenceno;
  int nberrors;

  wchart_chunker(ostream & os, ostream & err, const string & axiom_, bool verb)
    : out(os), errors(err), axiom(axiom_), verbose(verb), sentenceno(0), nberrors(0) {}

  void operator()(const wrtn_chart & chart, int res) {

    out << "#" << sentenceno << " " << chart.fsa.text << "\n\n";

    if (res == -1) { // parse too long
      cerr << "errror with sentence " << sentenceno << " : too much time\n";
      out << "ERROR\n";
      errors << chart.fsa.text << "\n\n";
      ++nberrors;
    }

    int n = output_matches(chart, axiom, out);

    if (n) {
      out << "." << n << " analyse(s).\n";
    } else {
      out << "FAIL\n";
    }

    if (verbose) {
      cout << n << " analyses\n\n";
    }
    ++sentenceno;

    if ((! verbose) && (sentenceno % 500) == 0) {
      cout << sentenceno << " sentences ...\n";
    }
  }
};


} // namespace ""



int main(int argc, char ** argv) try {

  fs::path txtpath, lingdefpath, grampath, opath;
  fs::path errorpath("chunk-errors");

  int PARSER_FLAGS = 0;// = wrtn_parser::IGNORE_DIFF_PATHS;
  double maxtime = 15.;

  bool verbose = false;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }


  argv++, argc--;

  if (! argc) { usage(); }

  while (argc) {
    
    string arg = *argv;
    
    if (arg == "-l") {
      
      argv++, argc--;
      if (argc == 0) { bad_args(); }
      lingdefpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-o") {
      
      argv++, argc--;
      if (argc == 0) { bad_args(); }
      opath = fs::path(*argv, fs::native);
    
    } else if (arg == "-gram") {
 
      argv++, argc--;
      if (argc == 0) { bad_args(); }
      grampath = fs::path(*argv, fs::native);
    
    } else if (arg == "-maxtime") {
 
      argv++, argc--;
      if (argc == 0) { bad_args(); }
      maxtime = lexical_cast<double>(*argv);
 
    } else if (arg == "-ipath") {
    
      PARSER_FLAGS |= wrtn_parser::IGNORE_DIFF_PATHS;

    } else if (arg == "-iout") {
    
      PARSER_FLAGS |= wrtn_parser::IGNORE_DIFF_OUTPUTS;

    } else if (arg == "-h") {
    
      usage();
    
    } else if (arg == "-v") {
    
      verbose = true;
    
    } else {

      txtpath = fs::path(arg, fs::native);
    }

    argv++, argc--;
  }

  if (txtpath.empty() || lingdefpath.empty() || grampath.empty()) { bad_args(); }

  if (opath.empty()) {
    opath = fs::change_extension(txtpath, ".chunks");
  }

  fs::ofstream out(opath);
  fs::ofstream errors(errorpath);

  ling_def lingdef(lingdefpath);

  scoped_ptr<itext_fsa> p_itext(new_itext_fsa(txtpath, & lingdef));
  itext_fsa & itext = *p_itext;

  wrtn_grammar gram(grampath, & lingdef);

  wchart_chunker chunker(out, errors, gram.start_name(), verbose);


  boost::timer tmr;

  wrtn_parse(itext, gram, chunker, PARSER_FLAGS, true, maxtime, verbose);

  cout << "\n" << chunker.sentenceno << " sentences parsed. " << chunker.nberrors << " errors. "
    << tmr.elapsed() << "s.\n";

  return 0;

} catch (exception & e) {

  cerr << "fatal error :" << e.what() << endl; exit(1);

} catch (...) { cerr << "caught an OVNI?\n"; exit(1); }

