#ifndef _SERIALIZE_H_
#define _SERIALIZE_H_

#include <iostream>
#include <cassert>


inline void write_int(std::ostream & os, int i) {
  os.put((i >> 24) & 0xff);
  os.put((i >> 16) & 0xff);
  os.put((i >>  8) & 0xff);
  os.put(i & 0xff);
}

inline void read_int(std::istream & is, int & n) {
  char c;
  n = 0;
  for (int i = 0; i < 4; i++) {
    is.get(c);
    n = (n << 8) | (c & 0xff);
  }
}


// return the number of bytes needed to serialize n

inline int n_bytes_needed(int n) {
  int res = 0;
  while (n) {
    ++res;
    n = n >> 8;
  }
  return res;
}


// serialize an integer in N bytes (big-endian encoding)

inline void write_intN(std::ostream & os, int n, int N) {
  while (N--) {
    os.put((n >> (N*8)) & 0xff); 
  }
}


// read an integer serialized in N bytes (big-endian encoding)

inline void read_intN(std::istream & is, int & n, int N) {
  char c;
  n = 0;
  while (N--) {
    is.get(c);
    n |= (c & 0xff) << (N*8);
  }
}

// mb is for multi-byte
// (litle-endian writing)

inline void write_int_mb(std::ostream & os, int n) {
  const int mask = 0x7f;
  while (n & ~mask) {
    os.put((n & mask) | 0x80);
    n = n >> 7;
  }
  os.put(n & mask);
}

inline void read_int_mb(std::istream & is, int & n) {
  const int mask = 0x7f;
  char c;
  int i = 0;
  n = 0;
  do {
    is.get(c);
    n |= (c & mask) << (7*i);
    ++i;
  } while (c & 0x80);
}


inline void write_int3(std::ostream & os, int i) {
  assert(i <= 0xffffff);
  os.put((i >> 16) & 0xff);
  os.put((i >>  8) & 0xff);
  os.put(i & 0xff);
}

inline void read_int3(std::istream & os, int & n) {
  char c;
  n = 0;
  os.get(c);
  n = (n << 8) | (c & 0xff);
  os.get(c);
  n = (n << 8) | (c & 0xff);
  os.get(c);
  n = (n << 8) | (c & 0xff);
}



inline void write_int2(std::ostream & os, int i) {
  assert(i <= 0xffff);
  os.put((i >>  8) & 0xff);
  os.put(i & 0xff);
}

inline void read_int2(std::istream & os, int & n) {
  char c;
  n = 0;
  os.get(c);
  n = (n << 8) | (c & 0xff);
  os.get(c);
  n = (n << 8) | (c & 0xff);
}


inline void write_ushort(std::ostream & os, unsigned short s) {
  os.put((s >> 8) & 0xff);
  os.put(s & 0xff);
}

inline void read_ushort(std::istream & os, unsigned short & s) {
  char c;
  s = 0;
  os.get(c);
  s = (s << 8) | (c & 0xff);
  os.get(c);
  s = (s << 8) | (c & 0xff);
}


inline void write_string(std::ostream & os, const std::string & str) {
  int size = str.size();
  //std::cerr << "write_string(...), size = " << size << std::endl;
  write_int_mb(os, size);
  os.write(str.c_str(), size);
}

inline void read_string(std::istream & is, std::string & str) {
  int size;
  read_int_mb(is, size);
  //std::cerr << "read_string() : size = " << size << std::endl;
  char txt[size];
  is.read(txt, size);
  str.assign(txt, size);
}

#endif

