#ifndef _RB_OLIVTREE_H_
#define _RB_OLIVTREE_H_

#include <iostream>
#include <stdexcept>

#include <boost/lexical_cast.hpp>

template<typename T>
class oliv_tree2 {

public:
  
  typedef T value_type;

  enum color_t { red, black };
  enum rb_code { ok, red_node, red_left, red_right };

  class node {

  public:

    color_t color;
    int card;

    value_type val;

    node * left, * right;

    node(const value_type & v, color_t col) : color(col), card(1), val(v), left(NULL), right(NULL) {}
  };


  oliv_tree2() : root(NULL) { /* std::cerr << "olivtree2\n"; */ }
  ~oliv_tree2() { clear(); }


  inline int size() const { return card(root); }

  void clear() { postorder(delete_node); }


  void insert(int idx, const value_type & val) {

    //  cerr << "insert : " << idx << '\n';
    //  check_integrity(); cerr << "ok\n";

    if (idx > size()) {
      std::string msg = "rbolivtree::insert: idx " + boost::lexical_cast<std::string>(idx)
        + " too big (size=" + boost::lexical_cast<std::string>(size()) + ")";
      std::cerr << msg << std::endl;
      throw std::runtime_error(msg);
    }

    switch (rec_insert(root, idx, val)) {
    case red_node:
      root->color = black;
    case ok:
      break;
    default:
      throw std::runtime_error("rb_insert : c'est n'importe quoi!\n");
    }

    /*
    std::cerr << "after insert: "; check_integrity(); cerr << "ok\n";
    assert((*this)[idx] == val);
    std::cerr << "insertion ok\n";
    */
  }


  const value_type & operator[] (int idx) const {
    const node * n = lookup_node(idx);
    if (n == NULL) { 
      throw std::runtime_error("olivtree[] const: idx " + boost::lexical_cast<std::string>(idx) + " not found"); 
    }
    return n->val;
  }

  value_type & operator[] (int idx) {
    node * n = lookup_node(idx);
    if (n == NULL) {
      std::cerr << "fatal: operator[]: idx=" << idx << " size=" << size() << ", idx not found\n";
      throw std::runtime_error("olivtree[]: idx not found");
    }
    return n->val;
  }

/*
  bool lookup(int key, value_type & val) const {
    const node * n = lookup_node(key);
    if (n == NULL) { return false; }
    val = n->val;
    return true;
  }
*/

  void check_integrity() {
    check_rb_condition(root, black);
    assert(check_card(root) == size());
  }

  template<typename Visitf>
  void preorder(Visitf visit) { rec_preorder(root, visit); }

  template<typename Visitf>
  void inorder(Visitf visit) { rec_inorder(root, visit); }

  template<typename Visitf>
  void postorder(Visitf visit) { rec_postorder(root, visit); }

  template<typename pre, typename in, typename pos>
  void travel();

  void dump(std::ostream & os) { dump(os, root); }

private:

  node * root;
  
  rb_code rec_insert(node * & n, int idx, const value_type & val) {

    if (n == NULL) {
//      assert(idx == 0);
      n = new node(val, red);
      return red_node;
    }

//    assert(idx <= n->card);

    rb_code code;
    int cardleft = card(n->left);

    if (idx <= cardleft) { // insert a gauche
      code = rec_insert(n->left, idx, val);
      n->card++;
      return modify_left(n, code);
    }

    // idx > cardleft => insert at right

    code = rec_insert(n->right, idx - cardleft - 1, val);
    n->card++;

    return modify_right(n, code);
  }


  rb_code modify_left(node * & n, rb_code code) {
    
    //std::cerr << "modifyleft: " << code << endl;

    if (code == ok) { return ok; }

    if (code == red_node) {
      if (n->color == black) { return ok; }
      return red_left;
    }
  
    //assert(n->color == black);
    color_t aunt_color = (n->right) ? n->right->color : black;

    if (aunt_color == red) { // si la tante est rouge aussi on change les couleurs
      flip_color(n);
      return red_node;
    }
 
    // uncle is black

    if (code == red_left) { // left left red condition
     rotate_right(n);
     n->color = black;
     n->left->color = red;
     n->right->color = red;
     return ok;
    }
 
    // we have a left right condition 
//    assert(code == red_right);

    double_rotate_right(n);
    n->color = black;
    n->left->color = n->right->color = red;

    return ok;
  }



  void flip_color(node * & n) {
    n->left->color = black;
    n->right->color = black;
    n->color = red;
  }

  void rotate_right(node * & n) {
    node * dad = n->left;
    n->left = dad->right;
    dad->right = n;

    dad->card = n->card;
    n->card = card(n->left) + card(n->right) + 1;

    n = dad;
  }

  void double_rotate_right(node * & n) {
    
    node * dad   = n->left;
    node * child = dad->right;
    
    n->left    = child->right; // or child->left ???
    dad->right = child->left;  // or child->right ???

    child->left  = dad;
    child->right = n;

    child->card = n->card;
    n->card = card(n->left) + card(n->right) + 1;
    dad->card = card(dad->left) + card(dad->right) + 1;

    n = child;
  }


  rb_code modify_right(node * & n, rb_code code) {
 
    //cerr << "modifright: " << code << endl;

    if (code == ok) { return ok; }

    if (code == red_node) {
      if (n->color == black) { return ok; }
      return red_right;
    }
  

    color_t uncle_color = (n->left) ? n->left->color : black;

    if (uncle_color == red) { // si la tante est rouge aussi on change les couleurs
      flip_color(n);
      return red_node;
    }
 
    // uncle is black

    if (code == red_right) { // right right red condition

     rotate_left(n);
     n->color = black;
     n->left->color = n->right->color = red;
     return ok;
    }
 
    // we have a right left condition 
    //assert(code == red_left);

    double_rotate_left(n);
    n->color = black;
    n->left->color = n->right->color = red;

    return ok;
  }



  void rotate_left(node * & n) {
    node * dad = n->right;
    n->right = dad->left;
    dad->left = n;

    dad->card = n->card;
    n->card = card(n->left) + card(n->right) + 1;

    n = dad;
  }


  void double_rotate_left(node * & n) {

    node * dad   = n->right;
    node * child = dad->left;
 
    n->right  = child->left;   // or child->left ???
    dad->left = child->right;  // or child->right ???

    child->right = dad;
    child->left  = n;

    child->card = n->card;
    n->card     = card(n->right) + card(n->left) + 1;
    dad->card   = card(dad->right) + card(dad->left) + 1;

    n = child;
  }

  const node * lookup_node(int idx) const {
    if (idx >= size()) {
      std::cerr << "lookup_node: warning: idx (" << idx << ") too big! (size=" << size() << ")\n";
      return NULL;
    }
    return lookup_node(root, idx);
  }

  node * lookup_node(int idx) {
    if (idx >= size()) {
      std::cerr << "lookup_node: warning: idx (" << idx << ") too big! (size=" << size() << ")\n";
      return NULL;
    }
    return lookup_node(root, idx);
  }

  const node * lookup_node(const node * n, int idx) const {

    if (n == NULL) {return NULL; } // throw std::runtime_error("lookup_node: not found"); 

    int cardleft = card(n->left);

    if (idx < cardleft) { return lookup_node(n->left, idx); }
    if (idx > cardleft) { return lookup_node(n->right, idx - cardleft - 1); }
 
    // idx == cardleft
    return n;
  }

  node * lookup_node(node * n, int idx) {

    if (n == NULL) {
      std::cerr << "lookup_node: " << idx << " not found\n";
      return NULL;
      // throw std::runtime_error("lookup_node: not found");
    }

    int cardleft = card(n->left);

    if (idx < cardleft) { return lookup_node(n->left, idx); }
    if (idx > cardleft) { return lookup_node(n->right, idx - cardleft - 1); }
 
    // idx == cardleft
    return n;
  }



  int check_rb_condition(node * n, color_t dad_color) const {
    if (n == NULL) { return 0; }
    if (dad_color == red) { assert(n->color == black); }
    int nblack = check_rb_condition(n->left, n->color);
    assert(nblack == check_rb_condition(n->right, n->color));
    return nblack + ((n->color == black) ? 1 : 0);
  }


  int check_card(node * n) const {

    if (n == NULL) { return 0; }
 
    assert(check_card(n->left) + check_card(n->right) + 1 == n->card);

    return n->card; 
  }



  template<typename Visitf>
  void rec_preorder(node * & n, Visitf visit) {
    if (n != NULL) {
      visit(n);
      rec_preorder(n->left, visit);
      rec_preorder(n->right, visit);
    }
  }
  template<typename Visitf>
  void rec_inorder(node * & n, Visitf visit) {
    if (n != NULL) {
      rec_inorder(n->left, visit);
      visit(n);
      rec_inorder(n->right, visit);
    }
  }
  template<typename Visitf>
  void rec_postorder(node * & n, Visitf visit) {
    if (n != NULL) {
      rec_postorder(n->left, visit);
      rec_postorder(n->right, visit);
      visit(n);
    }
  }


  void dump(std::ostream & os, const node * n) const {
    if (n == NULL) { return; }
    os << "( ";
    dump(os, n->left);
    os << " [" << n->card << "] ";
    dump(os, n->right);
    os << ") ";
  }

  static inline int card(const node * n) { return (n == NULL) ? 0 : n->card; }

  static void delete_node(node * & n) { delete n; n = NULL; }
};

#endif
