#ifndef _LEXICO_TREE_H_
#define _LEXICO_TREE_H_

#include <map>
#include <iterator>


template<typename ValueType = char>
class lexico_tree {

public:

  class node;
  typedef ValueType value_type;

  typedef std::map<value_type, node> translist;

  class node {
  public:
    int no;
    translist trans;

    inline node() : no(-1), trans() {}
    inline ~node() {}

    template<typename Functor>
    void visit(std::vector<value_type> & v, Functor & f) const {

      if (no != -1) { f(v); }

      typename translist::const_iterator it;

      for (it = trans.begin(); it != trans.end(); ++it) {
        v.push_back((*it).first);
        (*it).second.visit(v, f);
        v.pop_back();
      }
    }

    void dump(std::vector<value_type> & v, std::ostream & os) const { 

      if (no != -1) {
        os << no << ": ";
        std::copy(v.begin(), v.end(), std::ostream_iterator<value_type>(os, " "));
        os << "\n";
      }

      typename translist::const_iterator it;

      for (it = trans.begin(); it != trans.end(); ++it) {
        v.push_back((*it).first);
        (*it).second.dump(v, os);
        v.pop_back();
      }
    }

    inline void clear() { trans.clear(); no = -1; }

    void swap(node & n) { std::swap(no, n.no); trans.swap(n.trans); }
  };

  node root;


  template<typename Iterator>
  node * insert_path(Iterator first, Iterator last) {

    node * tree = & root;

    while (first != last) {

      typename translist::iterator it = (*tree).trans.find(*first);

      if (it == tree->trans.end()) {
        it = tree->trans.insert(tree->trans.begin(), make_pair(*first, node()));
      }

      tree = & (*it).second;
      ++first;
    }

    return tree;
  }

public:


  inline lexico_tree() : root() {}
  inline ~lexico_tree() {}



  template<typename Iterator>
  inline void insert(Iterator first, Iterator last, int idx = 1) {
    node * n = insert_path(first, last);
    n->no = idx;
  }

  template<typename Container>
  inline void insert(const Container & c, int no = 1) { insert(c.begin(), c.end(), no); }


  template<typename Iterator>
  inline int add_if_not_here(Iterator first, Iterator last, int idx) {
    node * n = insert_path(first, last);
    if (n->no == -1) { n->no = idx; }
    return n->no;
  }

  template<typename Container>
  inline int add_if_not_here(const Container & c, int idx) { return add_if_not_here(c.begin(), c.end(), idx); }
 

  template<typename Iterator>
  const node * lookup_path(Iterator first, Iterator last) const {
 
    const node * tree = & root;

    while (first != last) {

      typename translist::const_iterator it = (*tree).trans.find(*first);

      if (it == tree->trans.end()) {
        return NULL;
      }

      tree = & (*it).second;
      ++first;
    }

    return tree;
  }

  template<typename Container>
  const node * lookup_path(const Container & c) const { return lookup_path(c.begin(), c.end()); }

  template<typename Iterator>
  inline int lookup_idx(Iterator first, Iterator end) const {
    const node * n = lookup_path(first, end);
    if (n) { return n->no; }
    return -1;
  }

  template<typename Container>
  inline int lookup_idx(const Container & c) const { return lookup_idx(c.begin(), c.end()); }

  inline void clear() { root.clear(); }

  template<typename Functor>
  void visit(Functor & f) const { 
    std::vector<value_type> v;
    root.visit(v, f);
  }

  void dump(std::ostream & os) const { 
    std::vector<value_type> v;
    root.dump(v, os);
  }

  void swap(lexico_tree<value_type> t) { root.swap(t.root); }
};


#endif

