#ifndef _LOAD_FST2_H_
#define _LOAD_FST2_H_

#include <iostream>
#include <sstream>
#include <vector>
#include <stdexcept>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/fstream.hpp>



template<typename SymbolType, typename LoadFST2SymbolF>
void load_FST2_symbols(const boost::filesystem::path & fstpath, std::vector<SymbolType> & symbols,
                            LoadFST2SymbolF & load_FST2_symbol) {

  symbols.clear();

  boost::filesystem::ifstream is(fstpath);

  if (! is) {
    throw std::runtime_error("load_FST_symbols : unable to open " + fstpath.native_file_string());
  }

  try {

    is.exceptions(std::ios::badbit | std::ios::failbit);

    int nbsentences;
    is >> nbsentences;
    
    if (nbsentences == 0) { return; }

    std::string line;

    int i = 0;
    while (i < nbsentences) {
      std::getline(is, line);
      if (! line.empty() && line[0] == 'f') { i++; }
    }

    int size = 0;

    while (std::getline(is, line) && line[0] != 'f') {
      assert(line[0] == '%');
      if (line[line.size() - 1] == '\r') { line.resize(line.size() - 1); }
      symbols.resize(size + 1);
      load_FST2_symbol(line.substr(1), symbols[size]);
      //std::cerr << "symbol " << line << " loaded, type=" << symbols[size].type << '\n';
      size++;
    }

  } catch (std::ios::failure & e) {
    throw std::runtime_error("FST_symbol: parsing error in " + fstpath.native_file_string());
  }

 // std::cerr << "load_FST2_symbols: " << symbols.size() << " symbols loaded\n";
}


template<typename FstType, typename Symbol, typename AddTransF>
void load_FST2(std::istream & is, FstType & fst, std::vector<Symbol> & FST2_symbols,
               AddTransF add_transition) {

  typedef Symbol  symbol_type;
  typedef FstType fst_type;

  fst.clear();

  try {

    is.exceptions(std::ios::badbit | std::ios::failbit);

    std::string line, flags;

    getline(is, line);
    if (line.empty() || line[0] != '-') {
      throw std::runtime_error("synt_fst_from_FST: parse error with line " + line);
    }

    while (1) {

      if (! getline(is, line) || line.empty()) {
        throw std::runtime_error("parse_FST2: bad file format line=" + line);
      }
      if (line[0] == 'f') { break; }

      int q = fst.add_state();

//      cerr << "new state: " << q << "\n";
 
      std::istringstream iss(line);
      iss >> flags;

      bool final = (flags == "t") ? true : false;

      fst.set_final(q, final);


      int label, dest;
      while (iss >> label) {

        //       std::cerr << "new transition : ";

        iss >> dest;
        add_transition(fst, q, FST2_symbols[label], dest);
      }
    }

  } catch (std::ios::failure & e) {
    throw std::runtime_error("synt_fst_from_FST2: parsing error in FST2 file");
  }

//  cerr << "FST2 parsed: (" << fst.size() << " states).\n";
}


template<typename FstType, typename LoadFST2SymbolF, typename AddTransF>
void load_FST2(const boost::filesystem::path & fstpath, FstType & fst,
               LoadFST2SymbolF & load_FST2_symbol, AddTransF add_transition) {

  typedef FstType fst_type;
  typedef typename LoadFST2SymbolF::symbol_type symbol_type;

  std::vector<symbol_type> FST_symbols;

  load_FST2_symbols(fstpath, FST_symbols, load_FST2_symbol);

  boost::filesystem::ifstream is(fstpath);

  if (! is) {
    throw std::runtime_error("load FST2: cannot open " + fstpath.native_file_string());
  }


  std::string line;
  getline(is, line);

  int nbfst;
  std::istringstream(line) >> nbfst;

  if (nbfst != 1) {
    std::cerr << "warning: load_FST2: file contains " << nbfst << " FSTs, loading the first one\n";
  }

  load_FST2(is, fst, FST_symbols, add_transition);
}

#endif
