#include <string>
#include <boost/filesystem/convenience.hpp>

#include <outilex/syntagm_pattern.h>


#include <outilex/FST2.h>
#include <outilex/FST2_to_syntagm_pattern.h>

using namespace std;
namespace fs = boost::filesystem;

/*
typedef
struct synt_FST2_symbol<
  syntagm_pattern::input_type,
  syntagm_pattern::output_type,
  syntagm_pattern::synt_input_type,
  syntagm_pattern::synt_output_type>
pattern_symbol;
*/

namespace {

struct pattern_symbol {

  typedef syntagm_pattern::input_type      input_type;
  typedef syntagm_pattern::output_type     output_type;
  typedef syntagm_pattern::synt_input_type  synt_input_type;
  typedef syntagm_pattern::synt_output_type synt_output_type;

  enum { LEX, SYNT, BAD } type;

  input_type       in;
  output_type      out;
  synt_input_type  synt_in;
  synt_output_type synt_out;
};

struct FST2_pattern_symbol_loader {
  
  typedef pattern_symbol symbol_type;

  ling_def * lingdef;
  FST2_pattern_symbol_loader(ling_def * ldef) : lingdef(ldef) {}

  void operator()(const string & text, pattern_symbol & symb) {

    if (text.empty()) { symb.type = pattern_symbol::BAD; return; }

    string::size_type slash = text.find('/');
    string in, out;

    if (slash != string::npos) {
      in = text.substr(0, slash);
      out = text.substr(slash + 1);
    } else {
      in = text;
    }

    // parse input label

    try {

      if (in.compare(0, 2, "<:", 2) == 0) { // if label start with '<:' it is a syntagm mask

        symb.synt_in.read_text(in, lingdef);

        if (out.empty()) {
          symb.synt_out = symb.synt_out.one();
        } else {
          symb.synt_out.read_text(out);
        }

        symb.type = pattern_symbol::SYNT;

      } else { // else assume lexical mask

        symb.in.read_text(in, lingdef);

        if (out.empty()) {
          symb.out = symb.synt_out.one();
        } else {
          symb.out.read_text(out);
        }

        symb.type = pattern_symbol::LEX;
      }

    } catch (exception & e) {
      cerr << "error cannot load FST symbol '" << text << "': " << e.what() << "\n";
      symb.type = pattern_symbol::BAD;
    }
  }
};


void pattern_add_transition(syntagm_pattern & fst, int q, const pattern_symbol & symb, int dest) {
  if (symb.type == pattern_symbol::LEX) {
//    std::cerr << " lex trans\n";
    fst.add_trans(q, symb.in, symb.out, dest);
  } else if (symb.type == pattern_symbol::SYNT) {
//    std::cerr << " synt trans\n";
    fst.add_synt_trans(q, symb.synt_in, symb.synt_out, dest);
  } else {
    std::cerr << "bad transition label: type=" << symb.type << "\n";
  }
}

}; // anonymous namespace 


void FST2_to_syntagm_pattern(const fs::path & fstpath, const string & name,
                             syntagm_pattern & pattern) {
  pattern.clear();
  pattern.set_name(name);

  FST2_pattern_symbol_loader symbol_loader(pattern.get_lingdef());
  load_FST2(fstpath, pattern, symbol_loader, pattern_add_transition);
}

void FST2_to_syntagm_pattern(const fs::path & fstpath, syntagm_pattern & pattern) {

  fs::path p = fs::change_extension(fstpath, "");
  string name = p.leaf();

  pattern.clear();
  pattern.set_name(name);

  FST2_pattern_symbol_loader symbol_loader(pattern.get_lingdef());
  load_FST2(fstpath, pattern, symbol_loader, pattern_add_transition);
}


void FST2_to_syntagm_pattern(const fs::path & fstpath, const fs::path & patternpath,
                             const string & name, ling_def * lingdef) {

  syntagm_pattern pattern(lingdef);

  FST2_to_syntagm_pattern(fstpath, name, pattern);

  pattern.write_XML(patternpath);
}


void FST2_to_syntagm_pattern(const fs::path & fstpath, const fs::path & patternpath,
                             ling_def * lingdef) {

  fs::path p = fs::change_extension(fstpath, "");
  string name = p.leaf();

  syntagm_pattern pattern(lingdef);

  FST2_to_syntagm_pattern(fstpath, name, pattern);

  pattern.write_XML(patternpath);
}

