#include <string>

#include <boost/lexical_cast.hpp>

#include <outilex/xml-names.h>
#include <outilex/xml_format_uchart.h>
#include <outilex/xml_format_featstruct.h>
#include <outilex/xml_inl_format_sentence_fsa.h>

#include <outilex/uchart.h>

using namespace std;
using namespace boost;
using namespace xmlnames;

namespace {

void node_write_xml(xmlwriter & writer, const uchart & chart, int q) {

  writer.start_element(NODE_ELEM);
  writer.write_attribute(ID_ATTR, lexical_cast<string>(q));

  uchart::const_synt_iterator it = chart.synt_begin(q), end = chart.synt_end(q);
  for (int id = 1; it != end; ++it, ++id) {
    syntagm_write_xml(writer, *it, id);
  }
  writer.end_element();
}

void node_read_xml(xmlNode * node, uchart & chart, int no, ling_def * ldef) {

  node = node->children;
  while (node) {
    if (xmlStrcmp(node->name, SYNTAGM_ELEM) == 0) {
      syntagm synt;
      syntagm_read_xml(node, synt, ldef);
      chart.add_match(no, synt);
    }
    node = node->next;
  }
}

void synt_table_read_xml(xmlNode * node, uchart & chart, ling_def * ldef) {

  char * text = xmlGetConstProp(node, SIZE_ATTR);
  if (text) {
    int size = lexical_cast<int>(text);
    chart.synt_table.reserve(size);
  }

  int qno = 0;
  node = node->children;
  while (node) {
    if (xmlStrcmp(node->name, NODE_ELEM) == 0) {
      chart.synt_table.resize(qno + 1);
      node_read_xml(node, chart, qno, ldef);
      ++qno;
    }
    node = node->next;
  }
}

void synt_table_write_xml(xmlwriter & writer, const uchart & chart) {

  int size = chart.synt_table.size();

  writer.start_element(TABLE_ELEM);
  writer.write_attribute(SIZE_ATTR, lexical_cast<string>(size));

  for (int i = 0; i < size; ++i) {
    node_write_xml(writer, chart, i);
  }
  writer.end_element(); // table
}

} // namespace ""


void syntagm_write_xml(xmlwriter & writer, const syntagm & synt, int id) {
  writer.start_element(SYNTAGM_ELEM);

  if (id != -1) {
    writer.write_attribute(ID_ATTR, lexical_cast<string>(id));
  }
  writer.write_attribute(NAME_ATTR, synt.name);
  writer.write_attribute(TO_ATTR, lexical_cast<string>(synt.to));
  writer.write_attribute(WEIGHT_ATTR, lexical_cast<string>(synt.w));

  writer.start_attribute(PATH_ATTR);
  synt_path_type::const_iterator it = synt.path.begin(), end = synt.path.end();
  for (; it != end; ++it) {
    writer.write_string(lexical_cast<string>(it->qno) + " " + lexical_cast<string>(it->transno) + " ");
  }
  writer.end_attribute();

  featstruct_write_xml(writer, synt.fs);
  writer.end_element();
}

void syntagm_read_xml(xmlNode * node, syntagm & synt, ling_def * ldef) {

  synt.name = xmlGetConstProp(node, NAME_ATTR);

  synt.to = lexical_cast<int>(xmlGetConstProp(node, TO_ATTR));

  synt.w = lexical_cast<double>(xmlGetConstProp(node, WEIGHT_ATTR));

  istringstream is(xmlGetConstProp(node, PATH_ATTR));

  int qno, transno;
  while (is >> qno) {
    is >> transno;
    synt.path.push_back(syntref(qno, transno));
  }

  node = node->children;
  while (node) {
    if (xmlStrcmp(node->name, FEATSTRUCT_ELEM) == 0) {
      featstruct_read_xml(node, synt.fs, ldef);
    }
    node = node->next;
  }
}


void uchart_write_xml(xmlwriter & writer, const uchart & chart) {

  writer.start_element(UCHART_ELEM);

  sentence_fsa_write_inline_xml(writer, chart.fsa);

  synt_table_write_xml(writer, chart);

  writer.end_element(); // uchart
}


void uchart_read_xml(xmlNode * node, uchart & chart, ling_def * ldef) {


  chart.clear();

  node = node->children;
  while (node) {

    if (xmlStrcmp(node->name, SENTENCE_FSA_ELEM) == 0) {
    
      sentence_fsa_read_inline_xml(node, chart.fsa, ldef);
    
    } else if (xmlStrcmp(node->name, TABLE_ELEM) == 0) {

      synt_table_read_xml(node, chart, ldef);
    }

    node = node->next;
  }
}


