#include <string>
#include <boost/lexical_cast.hpp>
#include <outilex/xml_format_featstruct.h>

#include <outilex/xml.h>
#include <outilex/xml-names.h>
#include <outilex/featstruct.h>

using namespace std;
using namespace boost;
using namespace xmlnames;

void featstruct_write_xml(xmlwriter & writer, const featstruct & fs) {
  fs.write_xml(writer);
}

void featstruct_read_xml(xmlNode * node, featstruct & fs, ling_def * lingdef) {
  fs.read_xml(node, lingdef);
}


void featstruct::write_xml(xmlwriter & writer, int a, int & count, int * tag) const {

  a = follow(a);

  if (tag[a] < 0) { // node already processed, make a ref
    writer.start_element(FS_ELEM);
    writer.write_attribute(REF_ATTR, lexical_cast<string>(-tag[a]));
    writer.end_element();
    return;
  }

  writer.start_element(FS_ELEM);

  if (tag[a] > 1) { // first pass of a multipass node, print id
    writer.write_attribute(ID_ATTR, lexical_cast<string>(count));
    tag[a] = -count;
    ++count;
  }

  switch (core[a].type) {
  
  case FS_SET: {
    writer.write_attribute(TYPE_ATTR, "SET");
    int elem = a + core[a].next;
    while (elem != a) {
      writer.start_element(FEAT_ELEM);
      write_xml(writer, elem + core[elem].offset, count, tag);
      writer.end_element();
      elem += core[elem].next;
    }
  }
    break;

  case FS_FS: {
    writer.write_attribute(TYPE_ATTR, "complex");
    int attr = a + core[a].next;
    while (attr != a) {
      writer.start_element(FEAT_ELEM);
      writer.write_attribute(NAME_ATTR, core[attr].str);
      write_xml(writer, attr + core[attr].offset, count, tag);
      writer.end_element();
      attr += core[attr].next;
    }
  }
    break;

  case FS_VAL:
    writer.write_attribute(TYPE_ATTR, "feat");
    core[a].feat.write_XML(writer);
    break;

  case FS_CONSTR:
    writer.write_attribute(TYPE_ATTR, "constr");
    writer.write_string(core[a].str);
    break;

  case FS_STRING:
    writer.write_attribute(TYPE_ATTR, "str");
    writer.write_string(core[a].str);
    break;

  case FS_NOEXIST:
    writer.write_attribute(TYPE_ATTR, "noexist");
    break;

  case FS_EXIST:
    writer.write_attribute(TYPE_ATTR, "exist");
    break;

  case FS_UNSET:
    writer.write_attribute(TYPE_ATTR, "unset");
    break;

  default:
    throw logic_error("fs::write_xml: bad fs: invalid type : " + lexical_cast<string>(core[a].type));
    break;
  }

  writer.end_element(); // FS_ELEM
}


int featstruct::read_xml_rec(xmlNode * n, vector<int> & tag, ling_def * ldef) {

  int res;

  char * text = xmlGetConstProp(n, REF_ATTR);
  if (text) { // ref node
    res = new_node(FS_REF);  
    core[res].offset = tag[lexical_cast<int>(text)] - res;
    return res;
  }

  res = new_node(FS_UNSET);

  text = xmlGetConstProp(n, ID_ATTR);
  if (text) { // id attribut (reentrancy node)
    int id = lexical_cast<int>(text);
    if (id >= tag.size()) { tag.resize(id + 1); }
    tag[id] = res;
  }

  text = xmlGetConstProp(n, TYPE_ATTR);

  if (! text) {
    throw runtime_error("fs::read_xml: bad xml file (fs without type attribute)");
  }

  if (xmlStrcmp(text, "SET") == 0) {

    core[res].type = FS_SET;
    int curr = res;
    int elem;

    n = n->children;
    while (n) {
      if (xmlStrcmp(n->name, FEAT_ELEM) == 0) {
      
        elem = new_node(FS_SETELEM);
        
        core[elem].next = res - elem;
        core[curr].next = elem - curr;

        xmlNode * n2 = n->children;
        while (n2) {
          if (xmlStrcmp(n2->name, FS_ELEM) == 0) {
            int v = read_xml_rec(n2, tag, ldef);
            core[elem].offset = v - elem;
            break;
          }
          n2 = n2->next;
        }
        curr = elem;
      }
      n = n->next;
    }

  } else if (xmlStrcmp(text, "complex") == 0) {

    core[res].type = FS_FS;
    int curr = res, attr;

    n = n->children;
    while (n) {

      if (xmlStrcmp(n->name, FEAT_ELEM) == 0) {

        attr = new_node(FS_ATTR);
        core[attr].next = res - attr;
        core[attr].str = xmlGetConstProp(n, NAME_ATTR);

        core[curr].next = attr - curr;
 
        xmlNode * n2 = n->children;
        while (n2) {
          if (xmlStrcmp(n2->name, FS_ELEM) == 0) {
            int v = read_xml_rec(n2, tag, ldef);
            core[attr].offset = v - attr;
            break;
          }
          n2 = n2->next;
        }
        curr = attr;
      }
      n = n->next;
    }
 
  } else if (xmlStrcmp(text, "feat") == 0) {
  
    core[res].type = FS_VAL;
    n = n->children;
    while (n) {
      if (xmlStrcmp(n->name, feat_set::xml_name()) == 0) {
        core[res].feat.read_XML(n, ldef);
        break;
      }
      n = n->next;
    }

  } else if (xmlStrcmp(text, "str") == 0) {
  
    core[res].type = FS_STRING;
    text = (char *) xmlNodeGetContent(n);
    core[res].str = text;
    xmlFree(text);
  
  } else if (xmlStrcmp(text, "constr") == 0) {
  
    core[res].type = FS_CONSTR;
    text = (char *) xmlNodeGetContent(n);
    core[res].str = text;
    xmlFree(text);
  
  } else if (xmlStrcmp(text, "exist") == 0) {
  
    core[res].type = FS_EXIST;
  
  } else if (xmlStrcmp(text, "noexist") == 0) {
  
    core[res].type = FS_NOEXIST;

  } else if (xmlStrcmp(text, "unset") == 0) {
  
    core[res].type = FS_UNSET;
  
  } else {

    throw runtime_error("fs:read_xml: unknow fs node type: "+ string(text));
  }

  return res;
}


void featstruct::write_xml(xmlwriter & writer) const {
  
  writer.start_element(FEATSTRUCT_ELEM);
  if (! *this) {
    writer.write_attribute(BOTTOM_ATTR, "1");
    writer.end_element();
    return;
  }

  int tag[size()];

  memset(tag, 0, size() * sizeof(int));
  make_tag(tag);
  int count = 1;
  write_xml(writer, entry, count, tag);

  writer.end_element(); // FEATSTRUCT_ELEM
}


void featstruct::read_xml(xmlNode * node, ling_def * ldef) {

  char * text = xmlGetConstProp(node, BOTTOM_ATTR);

  if (text && text[0] == '1') {
    set_bottom();
    return;
  }

  core.clear();

  vector<int> tag;

  node = node->children;
  while (node) {
    if (xmlStrcmp(node->name, FS_ELEM) == 0) {
      entry = read_xml_rec(node, tag, ldef);
      break;
    }
    node = node->next;
  }
}


#if 0
void featstruct::write_XML(xmlwriter & writer) const {
  
  writer.start_element(FEATSTRUCT_ELEM);
  
  if (! *this) { 
    writer.write_attribute(BOTTOM_ATTR, "1");
  } else {

    writer.write_attribute(ENTRY_ATTR, lexical_cast<string>(entry));
    writer.write_attribute(SIZE_ATTR, lexical_cast<string>(size()));

    for (int i = 0; i < size(); ++i) { 
      write_XML(writer, i);
    }
  }
 
  writer.end_element();
}


int featstruct::read_fsnode_XML(xmlNode * node, ling_def * ldef) {
  
  int res = -1;
  char * text = xmlGetProp(node, TYPE_ATTR);
  
  if (xmlStrcmp(text, "FS") == 0) {

    xmlFree(text);

    //    res = core->new_node(FS_FS, feats_map());
    //   feats_map & fs = core->feats(res);

    res = new_node(FS_FS);

    node = node->children;
    while (node) {
      if (xmlStrcmp(node->name, FEAT_ELEM) == 0) {

        int ref = lexical_cast<int>(xmlGetConstProp(node, REF_ATTR));
        int attr = add_attr(res, xmlGetConstProp(node, NAME_ATTR));
        core[attr].offset = ref - attr;
      }
      node = node->next;
    }

  } else if (xmlStrcmp(text, "VAL") == 0) {

    xmlFree(text);

    node = node->children;

    while (node && xmlStrcmp(node->name, feat_set::xml_name())) { node = node->next; }
    if (node == NULL) {
      throw xml_parse_error("featstruct::read_XML: invalid FS_VAL node");
    }

    //res = core->new_node(FS_VAL, feat_set());
    //core->feat(res).read_XML(node, ldef);
    res = new_node(FS_VAL);
    core[res].feat.read_XML(node, ldef);

  } else if (xmlStrcmp(text, "STRING") == 0) {
 
    xmlFree(text);
    text = (char *) xmlNodeGetContent(node);
    res = new_node(FS_STRING);
    core[res].str = text;
    xmlFree(text);
 
  } else if (xmlStrcmp(text, "UNSET") == 0) {
 
    xmlFree(text);
    res = new_node(FS_UNSET);
 
  } else if (xmlStrcmp(text, "REF") == 0) {

    xmlFree(text);
    text = xmlGetProp(node, REF_ATTR);
    int ref = lexical_cast<int>(text);
    xmlFree(text);
    res = new_node(FS_REF);
    core[res].offset = ref - res;

  } else {
    throw xml_parse_error("featstruct::read_XML: unknow fs node type" + string(text));
  }
  return res;
}


void featstruct::read_XML(xmlNode * node, ling_def * ldef) {

  char * text = xmlGetProp(node, BOTTOM_ATTR);
  if (text && text[0] == '1') {
    set_bottom();
    xmlFree(text);
    return;
  }
  xmlFree(text);

  core.clear();

  text = xmlGetProp(node, SIZE_ATTR);
  int size = lexical_cast<int>(text);
  xmlFree(text);

  core.reserve(size);

  text = xmlGetProp(node, ENTRY_ATTR);
  entry = lexical_cast<int>(text);
  xmlFree(text);

  node = node->children;
  while (node) {
    if (xmlStrcmp(node->name, FSNODE_ELEM) == 0) {
      read_fsnode_XML(node, ldef);
    }
    node = node->next;
  }

  if (core.size() != size) {
    cerr << "error: featstruct::read_XML: size mismatch!\n";
  }
}
#endif

