#include <iostream>
#include <boost/filesystem/path.hpp>
#include <boost/progress.hpp>
#include <boost/scoped_ptr.hpp>

#include <outilex/lingdef.h>
#include <outilex/text_fsa.h>

#include <outilex/ugrammar.h>
#include <outilex/xml_text_uchart.h>
#include <outilex/uparser_twopass.h>


using namespace std;
using namespace boost;

namespace fs = boost::filesystem;


namespace {

char * progname;

void usage() {
  cerr << "usage: " << progname << " -l <lingdef> -g <grammar> [ -o <out> ] <textfsa>\n";
  exit(1);
}
} // anonymous namespace


int main(int argc, char ** argv) try {

  fs::path itextpath, lingdefpath, grammarpath, opath;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }


  progname = *argv;
  argv++, argc--;

  while (argc) {

    string arg = *argv;

    if (arg == "-l") {

      argv++, argc--;
      if (argc == 0) { usage(); }
      lingdefpath = fs::path(*argv, fs::native);

    } else if (arg == "-o") {

      argv++, argc--;
      if (argc == 0) { usage(); }
      opath = fs::path(*argv, fs::native);

    } else if (arg == "-g") {

      argv++, argc--;
      if (argc == 0) { usage(); }
      grammarpath = fs::path(*argv, fs::native);

    } else if (arg == "-h") {

      usage();

    } else {
      itextpath = fs::path(arg, fs::native);
    }

    argv++, argc--;
  }

  if (itextpath.empty() || lingdefpath.empty() || grammarpath.empty()) {
    cerr << "missing arguments\n"; exit(1);
  }


  if (opath.empty()) {

    string leaf = itextpath.leaf();
    string::size_type dot = leaf.rfind('.');

    if (dot != string::npos && leaf.substr(dot) == ".gz") {
      leaf.erase(dot);
      dot = leaf.rfind('.');
    }
    if (dot != string::npos && leaf.substr(dot) == ".fsa") {
      leaf.erase(dot);
    }
    opath = itextpath.branch_path() / (leaf + ".chart");
  }


  ling_def lingdef(lingdefpath);

  scoped_ptr<itext_fsa> p_itext(new_itext_fsa(itextpath, & lingdef));
  itext_fsa & itext = *p_itext;

  int total = itext.size();

  xml_otext_uchart ochart(opath, total);

  ugrammar gram(grammarpath, & lingdef);

  uparser_twopass parser(gram);


  sentence_fsa fsa;
  uchart chart;

  int sentenceno = 0, nbmatches = 0;
  progress_display show_progress(total, cout);

  while (itext >> fsa) {

    chart.eat(fsa);
    parser.parse(chart);
    ochart << chart;
    
    nbmatches += chart[0].synt_by_name.count(gram.start_name());
    ++sentenceno, ++show_progress;
  }

  cout << "done. " << nbmatches << " matches.\n";

  return 0;

} catch (exception & e) {

  cerr << "caught an exception : " << e.what() << endl;
  exit(1);

} catch (...) { cerr << "ouch!\n"; exit(1); }

