#include <vector>
#include <string>
#include <stdexcept>

#include <outilex/stringtok.h>
#include <outilex/featstruct.h>
#include <outilex/ref_attr.h>

#include <outilex/unification.h>

using namespace std;


namespace {

ref_attr_type * string_attr;

}// namespace ""


void unification_init(ling_def * ldef) {
  string_attr = new ref_attr_type("__ustring__");
  ldef->add_attr_type(string_attr);
}


feat_set unify(const feat_set & a, const feat_set & b) try {

  if (a.type != b.type) { // diff attr_type, look if one is string type and upgrades...
  
    if (a.type == string_attr) {
      cerr << "unify upgrade from string!\n";
      string text = string_attr->get_val_set_text(a.val);
      feat_set na(b.type, text);
      return unify(na, b);
    
    } else if (b.type == string_attr) {
      cerr << "unify upgrade from string!\n";
      string text = string_attr->get_val_set_text(b.val);
      feat_set nb(a.type, text);
      return unify(a, nb);
    
    } else { // unification fail

      feat_set null(a.type);
      null.clear();
      assert(!null);
      return null;
    }
  }

  feat_set res(a);
  res &= b;

  return res;

} catch (exception & e) {

  cerr << "unify(featset, featset) error : " << e.what() << endl;
  feat_set res;
  res.clear();
  return res;
}


feat_set unify(const feat_set & a, const std::string & txt) try {
  
  feat_set res(a.type, txt);
  res.set_inter(a);
  return res;

} catch (exception & e) {
  cerr << "unify(featset, text) error : " << e.what() << endl;
  feat_set res;
  res.clear();
  return res;
}
 

// unify to char string: e.g. 'hum|conc' u '!conc!abst' = 'hum'

feat_set unify(const string & txta, const string & txtb) {
  //  cerr << "string unify : " << txta << " & " << txtb << endl;

  feat_set a(string_attr, txta), b(string_attr, txtb);
  return unify(a, b);

  /*
  feat_set res = unify(a, b);
  cerr << "res = " << res << endl;
  return res;
  return unify(a, b);
  */
}


bool check_constraints(const vector<string> & constraints, featstruct & fs) {
  
  //  cerr << "\ncheck constraints:\n";
  //  cerr << "fs = " << fs << endl;

  for (vector<string>::const_iterator it = constraints.begin(), end = constraints.end();
       fs && it != end; ++it) {

    const string & c = *it;
    // cerr << "constraint = " << c << endl;

    if (c.compare(0, 5, "debug", 5) == 0) {
      cerr << c << '\n';
      if (c.compare(0, 7, "debugfs", 7) == 0) {
        cerr << "-> " << fs << '\n';
      }
      continue;
    }

    string::size_type in;

    if (c.find('=') != string::npos) { // unification constraint (a=b=c..)

      vector<string> paths;
      stringtok(c, "=", back_inserter(paths));

      fs_path path(paths[0]);

      for (int i = 1; i < paths.size(); ++i) {
      
        const string & p = paths[i];

        if (p.empty()) {
          cerr << "check_constraints: bad constraint :" <<  c << ": empty operand\n";
          continue; 
        }
        
        int sz = p.size();

        if (p[0] == ':') { // equation contrainte (a=:'v')
        
          if (p[1] != '\'' || p[sz - 1] != '\'') {
            cerr << "check_constraints: bad constraints : " << c << endl;
            continue;
          }
        
          fs.flag_constraint(path, p.substr(2, sz - 3));

        } else if (p[0] == '\'') { // unify path with an atomic value (not a path)

          if (p[sz - 1] != '\'') {
            cerr << "check_constraints: bad constraint : " << c << endl; continue;
          }
          //cerr << "check_constraints: p=" << p << " substr=" << p.substr(1, c.size() - 2) << endl;
          fs.unify(path, p.substr(1, sz - 2));

        } else { // unify two pass in the feat struct
      
          fs.unify(path, fs_path(p));
        }
      }


      /* equation sur les ensembles
       * a<b ou 'a'<b
       */
    } else if ((in = c.find('<')) != string::npos) {

      string left = c.substr(0, in);
      fs_path setpath(c.substr(in + 1));

      if (left[0] == '\'') { // left operand is an atomic value
      
        if (left[left.size() - 1] != '\'') {
          cerr << "check_constraints : bad constraints: " << c << endl;
          continue;
        }

        fs.set_insert(setpath, left.substr(1, left.size() - 2));
 
      } else { // left operand is a feat path
      
        fs.set_insert(setpath, fs_path(left));
      }

    } else { // contrainte existentielle

      if (c[0] == '~') { // flag the path with NOEXIST

        fs.flag_noexist(c.substr(1));

      } else if (c[0] == '^') { // shortcut: ^number == number=$$.number

        fs_path p1(c.substr(1)), p2("$$." + c.substr(1));
        
        fs.unify(p1, p2);

      } else { // flag with EXIST

        fs.flag_exist(c);
      }
    }
  }
  //  cerr << "out of check-constraints: fs = " << fs << endl;
  return fs;
}



#if 0
en commentaire parceque inutile pour le moment

bool check_constraints(const vector<string> & constraints, list<featstruct> & fs) {

  list<featstruct>::iterator it = fs.begin();

  while (it != fs.end()) {
    if (! check_constraints(constraints, *it)) {
      it = fs.erase(it);
    } else {
      ++it;
    }
  }

  return ! fs.empty();
}


bool check_constraints(const std::vector<std::string> & constraints, std::list<featstruct> & fss,
                       const lexical_entry & matchval) {

  std::list<featstruct>::iterator it = fss.begin();

  while (it != fss.end()) {
    featstruct & fs = *it;
    fs.set("$$", matchval);
    if (! check_constraints(constraints, fs)) {
      it = fss.erase(it);
    } else {
      fs.unlink(fs_path(), "$$");
      ++it;
    }
  }

  return ! fss.empty();
}

bool check_constraints(const std::vector<std::string> & constraints, std::list<featstruct> & fss,
                       const featstruct & matchval) {

  /*
  cerr << "check_constraints:\n"; 
  copy(constraints.begin(), constraints.end(), ostream_iterator<string>(cerr, "\n"));

  cerr << "with:\n";
  list<featstruct>::iterator it = fss.begin();
  copy(fss.begin(), fss.end(), ostream_iterator<featstruct>(cerr, "\n"));
  cerr << "matchval = " << matchval << endl;
  */

  list<featstruct>::iterator it = fss.begin();

  while (it != fss.end()) {
    featstruct & fs = *it;
    fs.set("$$", matchval);
    if (! check_constraints(constraints, fs)) {
      //    cerr << "\nFAIL\n";
      it = fss.erase(it);
    } else {
      fs.unlink(fs_path(), "$$");
      //cerr << "\nSUCCESS: " << *it << endl;
      ++it;
    }
  }

  //cerr << "out of check_constraints: fss.size = " << fss.size() << "\n\n***\n\n";
  return ! fss.empty();
}
#endif
