#include <boost/lexical_cast.hpp>

#include <outilex/xml.h>
#include <outilex/xml-names.h>
#include <outilex/ugrammar.h>


using namespace std;
using namespace boost;
using namespace xmlnames;

namespace fs = boost::filesystem;


ugrammar::ugrammar(const fs::path & path, ling_def * ldef) 
  : lingdef(ldef), syntagms(), synt_map(), start_graph(0) {

  read(path);
}



int ugrammar::add_pattern(const usyntagm_pattern & pattern) {

  cerr << "warning: add_pattern: deep copy\n";

  int idx;
  const string & name = pattern.get_name();

  map<string, int>::iterator it = synt_map.find(name);

  if (it == synt_map.end()) {
  
    idx = syntagms.size();
    syntagms.resize(idx + 1);
  
    synt_map[name] = idx;

  } else { idx = it->second; }

  syntagms[idx] = pattern;
  return idx;
}


int ugrammar::add_pattern(const fs::path & patternpath, const std::string & name) {
  
  int idx;

  map<string, int>::iterator it = synt_map.find(name);
  
  if (it == synt_map.end()) { // new pattern
  
    idx = syntagms.size();
    syntagms.resize(idx + 1);

    synt_map[name] = idx;
  
  } else { // override
  
    idx = it->second;
  }

  syntagms[idx].read(patternpath, lingdef);
  //cerr << "syntagm[idx].name = " << syntagms[idx].get_name() << " & name = " << name << endl;
  assert(syntagms[idx].get_name() == name);

  if (syntagms[idx].empty()) {
    cerr << "warning: pattern " << name << " is empty\n";
    syntagms[idx].add_state();
  }
  return idx;
}



void ugrammar::write(const fs::path & path) const {

  xmlwriter writer(path);
  writer.set_indent(true);

  writer.start_document();
  write_XML(writer);
  writer.end_document();
}


void ugrammar::read(const fs::path & path) {

  string fname = path.native_file_string();

  xmlDocPtr doc = xmlParseFile(fname.c_str());

  if (doc == NULL) {
    throw xml_parse_error("ugrammar::read_XML: " + fname + " not successfully parsed");
  }

  xmlNodePtr node = xmlDocGetRootElement(doc);

  try {
    read_XML(node);
  } catch (exception & e) {
    cerr << "ugrammar::read: error while parsing grammar " << fname << ": " << e.what() << endl;
    xmlFreeDoc(doc);
    throw;
  }
  xmlFreeDoc(doc);
}


void ugrammar::write_XML(xmlwriter & writer) const {

  writer.start_element(UGRAMMAR_ELEM);

  writer.write_attribute(MAIN_ATTR, syntagms[start_graph].get_name());
  writer.write_attribute(SIZE_ATTR, lexical_cast<string>(syntagms.size()));

  for_each(syntagms.begin(), syntagms.end(), XML_Writer(writer));
}


void ugrammar::read_XML(xmlNodePtr node) {

  syntagms.clear();
  synt_map.clear();

  if (node == NULL) { throw xml_parse_error("ugrammar::read_XML: document is empty"); }

  if (xmlStrcmp(node->name, UGRAMMAR_ELEM)) {
    throw xml_parse_error("ugrammar:read_XML: wrong document type");
  }

  string mainname;

  char * text = xmlGetProp(node, MAIN_ATTR);
  if (text) {
    mainname = text;
    xmlFree(text);
  }

  text = xmlGetProp(node, SIZE_ATTR);
  if (text) {
    syntagms.reserve(lexical_cast<int>(text));
    xmlFree(text);
  }

  node = node->xmlChildrenNode;

  int syntno = 0;
  while (node) {

    if (xmlStrcmp(node->name, USYNTAGM_PATTERN_ELEM) == 0) {

      syntagms.resize(syntno + 1);
      syntagms[syntno].read_xml(node, lingdef);
      synt_map[syntagms[syntno].get_name()] = syntno;
      ++syntno;
    }
    node = node->next;
  }

  if (syntagms.empty()) {
    throw runtime_error("ugrammar::read xml: grammar is empty");
  }

  if (! mainname.empty()) {
    start_graph = synt_map[mainname];
  } else { start_graph = 0; }
}

