#include <algorithm>
#include <iterator>
#include <string>

#include <boost/shared_ptr.hpp>
#include <boost/lexical_cast.hpp>

#include <outilex/featstruct.h>
#include <outilex/unification.h>
#include <outilex/stringtok.h>
#include <outilex/lexical_entry.h>
#include <outilex/lexical_mask.h>
#include <outilex/fs_node.h>

using namespace std;
using namespace boost;



void fs_path::init(const std::string & p) {
  if (p == "$.") { // $. reference the root of the featstruct
    path.clear();
  } else {
    stringtok(p, ".", back_inserter(path));
  }
}

void fs_path::dump(ostream & os) const {
  copy(path.begin(), path.end() - 1, ostream_iterator<std::string>(os, "."));
  os << *(path.end() - 1);
}

std::string fs_path::string() const {
  ostringstream os;
  dump(os);
  return os.str();
}


#if 0
featstruct featstruct::clone() const {
  featstruct fs;
  fs.core.reset(new core_type(*core));
  fs.entry = entry;
  return fs;
}
#endif


void featstruct::clear() {
  core.clear(); // allocate a new empty core
  entry = setup_unset();
}


int featstruct::follow(int pos) const {
  while (core[pos].type == FS_REF) {
    pos += core[pos].offset;
  }
  return pos;
}

#if 0
int featstruct::follow(int pos) const {

  if (core->type(pos) != FS_REF) { return pos; }

  int & offst = core->offset(pos);
  int res = follow(pos + offst); 
  offst = res - pos;
  return res;
}
#endif

void featstruct::link(int ref, int pos) {
  core[ref].type = FS_REF;
  core[ref].offset = pos - ref;
}


int featstruct::add_attr(int attr, const std::string & attrname) {

  int attrnext = attr + core[attr].next;
  while (core[attrnext].type == FS_ATTR && core[attrnext].str < attrname) {
    attr = attrnext;
    attrnext += core[attrnext].next;
  }

  if (core[attrnext].type != FS_ATTR || attrname < core[attrnext].str) { // add new attr
    attrnext = setup_attr(attrname, attrnext);
    core[attr].next = attrnext - attr;
  }
  //assert(core->str(attrnext) == attrname);
  return attrnext;
}

void featstruct::add_elem(int set, int elem) {
  int pos = setup_elem(elem, set + core[set].next);
  core[set].next = pos - set;
}


int featstruct::setup_path(const fs_path & path) {

  int pos = follow(entry);

  for (fs_path::const_iterator it = path.begin(); it != path.end(); ++it) {

    if (core[pos].type == FS_UNSET) {

      core[pos].type = FS_FS;
      core[pos].next = 0; // end of list point to the root fs element

    } else if (core[pos].type != FS_FS) {
      cerr << "setup_path: invalid path " << path << "\n";
      return -1;
    }

    int attr = add_attr(pos, *it);
 
    if (core[attr].offset == 0) {
      pos = setup_unset();
      core[attr].offset = pos - attr;
    } else {
      pos = follow(attr + core[attr].offset);
    }
  }
  return pos;
}

int featstruct::setup_set(const fs_path & path) {

  int pos = setup_path(path);

  if (pos == -1) { return -1; }

  if (core[pos].type == FS_UNSET) {
    core[pos].type = FS_SET;
  }
  if (core[pos].type != FS_SET) { return -1; }
  return pos;
}

int featstruct::find_attr(int pos, const std::string & attrname) const {
  int attr = pos + core[pos].next;
  while (core[attr].type == FS_ATTR && core[attr].str < attrname) {
    attr = attr + core[attr].next;
  }

  if (core[attr].str == attrname) { // && core->type(attr) == ATTR
    return attr + core[attr].offset;
  }
  return -1;
}

int featstruct::follow(const fs_path & path, int pos) const {

  pos = follow(pos);

  for (fs_path::const_iterator it = path.begin(); it != path.end(); ++it) {

    if (core[pos].type != FS_FS) { return -1; }

    if ((pos = find_attr(pos, *it)) == -1) { return -1; }

    pos = follow(pos);
  }
  return pos;
}

void featstruct::unlink(const fs_path & path, const std::string & featname) {

  if (! *this) { return; }

  int pos = follow(path);
  if (pos == -1 || core[pos].type != FS_FS || core[pos].next == 0) {
    // path doesn't exist or doesn't point to a complex fs
    return;
  }

  int attr = pos;
  int attrnext = attr + core[attr].next;
  while (attrnext != pos && core[attrnext].str < featname) {
    attr = attrnext;
    attrnext += core[attrnext].next;
  }

  if (attrnext == pos || featname < core[attrnext].str) { // feat not present
    return;
  }

  /* attrnext point to the attr to remove */
  core[attr].next = attrnext + core[attrnext].next - attr;
}


bool featstruct::has_path(const fs_path & path) const {
  return follow(path) != -1;
}

int featstruct::get_type(const fs_path & path) const {
  int pos = follow(path);
  if (pos == -1) { return FS_UNSET; }
  return core[pos].type;
}

#if 0
const featstruct featstruct::get_fs_val(const fs_path & path) const {
  int pos = follow(path);
  if (pos == -1) { throw fs_bad_get(path, "path not found"); }
  featstruct res(*this);
  res.entry = pos;
  return res;
}

featstruct featstruct::get_fs_val(const fs_path & path) {
  int pos = follow(path);
  if (pos == -1) { throw fs_bad_get(path, "path not found"); }
  featstruct res(*this);
  res.entry = pos;
  return res;
}

const featstruct::feats_map & featstruct::get_fs_node_val(int pos) const {
  if (core->type(pos) != FS_FS) { throw fs_bad_get("not a complex FS"); }
  return core->feats(pos);
}
#endif


#if 0
const string & featstruct::get_string_val(int pos) const {
  if (core[pos].type != FS_STRING) { throw fs_bad_get("not a string"); }
  return core[pos].str;
}


const string & featstruct::get_string_val(const fs_path & path) const {
  int pos = follow(path);
  if (pos == -1) { throw fs_bad_get(path, "path not found"); }
  return get_string_val(pos);
}


const feat_set & featstruct::get_featset_val(int pos) const {
  if (core->type(pos) != FS_VAL) { throw fs_bad_get("not a string"); }
  return core->feat(pos);
}

const feat_set & featstruct::get_featset_val(const fs_path & path) const {
  int pos = follow(path);
  if (pos == -1) { throw fs_bad_get(path, "path not found"); }
  return get_featset_val(pos);
}
#endif


int featstruct::setup_attr(const std::string & attrname, int next) {
  string name(attrname); // make a local copy of attrname becose new_node can invalided it
  int res = new_node(FS_ATTR);
  core[res].str = name;
  core[res].next = next - res;
  return res;
}

int featstruct::setup_elem(int e, int next) {
  int res = new_node(FS_SETELEM);
  core[res].offset = e - res;
  core[res].next = next - res;
  return res;
}

int featstruct::setup_unset() {
  return new_node(FS_UNSET);
}

int featstruct::setup(const std::string & txt) {
  int res = new_node(FS_STRING);
  core[res].str = txt;
  return res;
}

int featstruct::setup(const feat_set & fs) {
  int res = new_node(FS_VAL);
  core[res].feat = fs;
  return res;
}


int featstruct::setup(const featstruct & fs) {

  if (! fs) { return -1; }
  
  int res = core.size();
  core.resize(res + fs.core.size());

  for (int i = 0; i < fs.core.size(); ++i) {
    core[res + i] = fs.core[i];
  }

  return res + fs.entry;
}


#if 0
int featstruct::setup(const lexical_entry & e) {

  //if (! *this) { return -1; }

  /* reserve enough space to avoid iterator invalidation
   * 1 : root node, 4 : form+lemma+pos+LEX
   *          (4+feats.size()) * 2 : because of 2 node for each attributes (attr node, value node)
   *                ...        * 3 just in case  ...
   */

  core.reserve(core.size() + 1 + (4 + e.feats.size()) * 3);

  //  int res = core->new_node(FS_FS, feats_map());
  //  feats_map & fs = core->feats(res);

  int root = new_node(FS_FS);

  //  int offset = setup("LEX");
  //  fs["CAT"] = offset - res;
 
  // add attributes in lexico graphical order
  int attr = add_attr(root, "CAT");
  core[attr].offset = setup("LEX") - attr;

  pos_def * pos = e.pos;
  attr = add_attr(attr, "POS");
  core[attr].offset = setup(pos->get_name()) - attr;

  attr = add_attr(root, "form");
  core[attr].offset = setup(e.form) - attr;

  attr = add_attr(attr, "lemma");
  core[attr].offset = setup(e.lemma) - attr;

  for (int i = 0; i < e.feats.size(); ++i) {
    if (e.feats[i] != 0) { // skip 'unset' values
      attr_def * attrd = pos->get_attr(i);
      attr = add_attr(root, attrd->get_name());
      core[attr].offset = setup(feat_set(attrd->get_type(), e.feats[i])) - attr;
    }
  }
  return root;
}
#endif

int featstruct::setup(const lexical_mask & m) {

  /* reserve enough space to avoid iterator invalidation
   * 1 : root, 4 : form+lemma+pos+"LEX"
   * etc.
   */

  core.reserve(core.size() + 1 + (4 + m.feats.size()) * 3);

  int root = new_node(FS_FS);

  // add attributes in lexico graphical order
  int attr = add_attr(root, "CAT");
  core[attr].offset = setup("LEX") - attr;

  const pos_def * pos = m.pos;
  attr = add_attr(attr, "POS");
  core[attr].offset = setup(pos->get_name()) - attr;

  attr = add_attr(root, "form");
  core[attr].offset = setup(m.form) - attr;

  attr = add_attr(attr, "lemma");
  core[attr].offset = setup(m.lemma) - attr;

  for (lexical_mask::feats_map::const_iterator it = m.feats.begin(), end = m.feats.end();
       it != end; ++it) {
    const attr_def * attrd = it->first;
    attr = add_attr(root, attrd->get_name());
    core[attr].offset = setup(it->second) - attr;
  }
  return root;
}



bool featstruct::set(const fs_path & path, const featstruct & fs) {
  
  if (! *this) { return false; }

  int pos = setup_path(path);
  int val = setup(fs);

  if (pos == -1 || val == -1) { set_bottom(); return false; }

  link(pos, val);
  return true;
}


bool featstruct::set(const fs_path & path, const lexical_mask & fs) {
  
  if (! *this) { return false; }

  int pos = setup_path(path);
  int val = setup(fs);

  if (pos == -1 || val == -1) { set_bottom(); return false; }

  link(pos, val);

  return true;
}

#if 0
bool featstruct::set(const fs_path & path, const lexical_entry & fs) {
  
  if (! *this) { return false; }

  int pos = setup_path(path);
  int val = setup(fs);

  if (pos == -1 || val == -1) { set_bottom(); return false; }

  link(pos, val);

  return true;
}
#endif

bool featstruct::set(const fs_path & path, const string & v) {
  
  if (! *this) { return false; }

  int pos = setup_path(path);
  if (pos == -1) { set_bottom(); return false; }

  core[pos].type = FS_STRING;
  core[pos].str = v;
  return true;
}


/* FLAGS stuffs */

bool featstruct::flag_exist(const fs_path & path) {

  if (! *this) { return false; }

  int pos = setup_path(path);
  if (pos == -1) { set_bottom(); return false; }

  switch (core[pos].type) {
  case FS_NOEXIST:
    set_bottom(); return false;
    break;
  case FS_UNSET:
    core[pos].type = FS_EXIST;
    break;
  case FS_EXIST:  // flag already present
  case FS_CONSTR: // CONSTR imply exist
  case FS_STRING: // value already exists
  case FS_VAL:
  case FS_FS:
    break;
  default:
    throw logic_error("fs::flag-exist: invalid feat type : " + lexical_cast<string>(core[pos].type));
    break;
  }
  return true;
}


bool featstruct::flag_noexist(const fs_path & path) {
  if (! *this) { return false; }

  int pos = setup_path(path);
  if (pos == -1) {
    cerr << "warning: set_noexit: bad path : " << path << endl;
    return true;
  } // bad path, return true ...

  if (core[pos].type == FS_UNSET || core[pos].type == FS_NOEXIST) {
    core[pos].type = FS_NOEXIST;
    return true;
  }
  // value already set
  set_bottom();
  return false;
}

bool featstruct::flag_constraint(const fs_path & path, const std::string & v) {

  if (! *this) { return false; }

  int pos = setup_path(path);
  if (pos == -1) { set_bottom(); return false; }

  switch (core[pos].type) {
  case FS_UNSET:
  case FS_EXIST:
    core[pos].type = FS_CONSTR;
    core[pos].str  = v;
    break;

  case FS_CONSTR:
  case FS_STRING:
    if (core[pos].str != v) {
      set_bottom(); return false;
    }
    break;

  case FS_VAL: {
    feat_set fs(core[pos].feat.type, v);
    if (fs != core[pos].feat) {
      set_bottom(); return false;
    }
  }
    break;
  
  default:// noexist, fs complex or bad type
    set_bottom(); return false;
    break;
  }
  return true;
}



bool featstruct::unify(const fs_path & p, const std::string & val) {

  if (! *this) { return false; }

  int pos1 = setup_path(p);
  int pos2 = setup(val);

  if ((pos1 == -1) || (pos2 == -1)) { // unification failed (bad path)
    set_bottom();
    return false;
  }
  return unify(pos1, pos2);
}

bool featstruct::unify(const fs_path & p1, const fs_path & p2) {

  if (! *this) { return false; }

  int pos1 = setup_path(p1);
  int pos2 = setup_path(p2);

  if ((pos1 == -1) || (pos2 == -1)) { // unification failed (bad path)
    set_bottom();
    return false;
  }
  return unify(pos1, pos2);
}


bool featstruct::unify(int a, int b) {

  if (unify_(a, b) == -1) { // unification failed
    set_bottom();
    return false;
  }
  return true;
}


int featstruct::unify_(int a, int b) {

  a = follow(a); b = follow(b);

  if (a == b) { return a; }

  if (core[a].type < core[b].type) { std::swap(a, b); }

  // type de b <= type de a
 
  if (core[b].type == FS_UNSET) {
    link(b, a);
    return a;
  }

  if (core[b].type == FS_EXIST) {
    if (core[a].type == FS_NOEXIST) { return -1; }
    link(b, a);
    return a;
  }

  if (core[b].type == FS_NOEXIST) {
    if (core[a].type != FS_NOEXIST) { return -1; }
    link(b, a);
    return a;
  }

  /* a partir d'ici : type(b) <= type(a) et type(b) != unset,exist,noexist
   */

  switch (core[a].type) {

  case FS_SET: {

    if (core[b].type != FS_SET) { return -1; }

    cerr << "warning unify SET: this is meaningless\n";

    // add each elem in b into a
    int elem = b + core[b].next;
    while (elem != b) {
      add_elem(a, elem + core[elem].offset);
    }
  }
  break;

  case FS_FS: {

    //cerr << "FS_FS\n";

    if (core[b].type != FS_FS) { return -1; }

    int attr1 = a,
        attr2 = b + core[b].next;
 
    while (core[attr2].type == FS_ATTR) { // for each attr in b
 
      attr1 = add_attr(attr1, core[attr2].str); // find or add attribute in a

      if (core[attr1].offset == 0) { // new attribute
 
        core[attr1].offset = attr2 + core[attr2].offset - attr1;
 
      } else { // unify feats

        int res = unify_(attr1 + core[attr1].offset, attr2 + core[attr2].offset);
        if (res == -1) { return -1; }
        core[attr1].offset = res - attr1;
        core[attr2].offset = res - attr2;
      }
      attr2 += core[attr2].next;
    }
    //cerr << "end of FS_FS\n";
  }
  break;

  case FS_VAL: {

    feat_set & fa = core[a].feat;

    if (core[b].type == FS_VAL) {

      feat_set & fb = core[b].feat;
      fa = ::unify(fa, fb);

    } else {
      // type(b) = STRING|CONSTR -> unify & return a feat
      fa = ::unify(fa, core[b].str);
    }
    if (! fa) { return -1; }
  }
  break;

  case FS_STRING:
    // type(b) == STRING|CONSTR -> unify & return a string

    if (core[a].str != core[b].str) {

      if (core[b].type != FS_STRING) { return -1; }

      if (core[a].str.find_first_of("!|") == string::npos
          && core[b].str.find_first_of("!|") == string::npos) {
          return -1;
      }

      // upgrade to feat and try something more subbtle as in 'm|f' u 'm' -> 'm' ?

      feat_set & f = core[a].feat;
      f = ::unify(core[a].str, core[b].str);
      if (! f) { return -1; }
      core[a].type = FS_VAL;
      core[a].str.clear();
    }
    break;

  default:
    throw logic_error("in unify invalid node type : " + lexical_cast<string>(core[a].type));

  }

  link(b, a);
  return a;
}


/// set stuffs


bool featstruct::set_insert(const fs_path & setpath, const fs_path & elempath) {

  int setpos = setup_set(setpath);
  int elempos = setup_path(elempath);

  if ((setpos == -1) || (elempos == -1)) { set_bottom(); return false; }

  add_elem(setpos, elempos); // never fails ?
  return true;
}


bool featstruct::set_insert(const fs_path & setpath, const string & val) {

  int setpos = setup_set(setpath);
  int elempos = setup(val);

  if ((setpos == -1) || (elempos == -1)) { set_bottom(); return false; }

  add_elem(setpos, elempos); // never fails ?
  return true;
}


/// compression

int featstruct::compress(int e, vector<node> & ncore, int * tag, bool cleandollar) {

  e = follow(e);
  
  if (tag[e] != -1) { // already passed here
    return tag[e];
  }

  int npos = -1;
  switch (core[e].type) {

  case FS_SET: {

    npos = ncore.size();
    ncore.resize(npos + 1);
    ncore[npos].type = FS_SET;
 
    int elem = e + core[e].next;
    int curr = npos;

    while (elem != e) {
    
      int nelem = ncore.size();
      ncore.resize(nelem + 1);
      ncore[nelem].type = FS_SETELEM;

      ncore[curr].next = nelem - curr;
      ncore[nelem].next = npos - nelem;
    
      int ne = compress(elem + core[elem].offset, ncore, tag, cleandollar);
      ncore[nelem].offset = ne - nelem;
      curr = nelem;
 
      elem += core[elem].next;
    }
  }
    break;
  
  case FS_FS: {

    npos = ncore.size();
    ncore.resize(npos + 1);
    ncore[npos].type = FS_FS;

    int attr = e + core[e].next, curr = npos;

    while (attr != e) {
      
      if (! cleandollar || core[attr].str[0] != '$') {
 
        int nattr = ncore.size();
        ncore.resize(nattr + 1);
        ncore[nattr].type = FS_ATTR;
        ncore[nattr].str = core[attr].str;

        ncore[curr].next  = nattr - curr;
        ncore[nattr].next = npos - nattr;

 
        int ne = compress(attr + core[attr].offset, ncore, tag, cleandollar);
        ncore[nattr].offset = ne - nattr;

        curr = nattr;
      }
      attr += core[attr].next;
    }
  }
    break;

  case FS_VAL:
    npos = ncore.size();
    ncore.resize(npos + 1);
    ncore[npos].type = FS_VAL;
    ncore[npos].feat = core[e].feat;
    break;

  case FS_STRING:
  case FS_CONSTR:
    npos = ncore.size();
    ncore.resize(npos + 1);
    ncore[npos].type = core[e].type;
    ncore[npos].str = core[e].str;
    break;

  case FS_UNSET:
  case FS_EXIST:
  case FS_NOEXIST:
    npos = ncore.size();
    ncore.resize(npos + 1);
    ncore[npos].type = core[e].type;
    break;

  default:
    throw logic_error("fs::compress: invalid node type : " + lexical_cast<string>(core[e].type));
    break;
  }

  tag[e] = npos;
  return npos;
}


void featstruct::compress(bool cleandollar) {

  if (! *this) { return; }

  int tag[core.size()];
  for (int i= 0; i < size(); ++i) { tag[i] = -1; }

  vector<node> ncore;
  ncore.reserve(size());

  entry = compress(entry, ncore, tag, cleandollar);
  core.swap(ncore);
}




void featstruct::prettyprint(int a, int & count, int * tag, int indent, ostream & os) const {

  a = follow(a);

  if (tag[a] < 0) { // node already passed, print reference
    os << '#' << -tag[a];
    return;
  }

  if (tag[a] > 1) { // first pass of a multipass node, print identifier
    ostringstream oss;
    oss << "(" << count << ") ";
    indent += oss.str().size();
    os << oss.str();
    tag[a] = -count;
    count++;
  }

  switch (core[a].type) {

  case FS_SET: {
  
    os << "{ ";
    indent +=2;
    
    int elem = a + core[a].next;
    
    if (elem != a) {
      prettyprint(elem + core[elem].offset, count, tag, indent, os);
      os << ", ";
      elem += core[elem].next;
    }
  
    while (elem != a) {
      
      os << '\n';
      for (int i = 0; i < indent; ++i) { os << ' '; }
      
      prettyprint(elem + core[elem].offset, count, tag, indent, os);
      os << ", ";
      elem += core[elem].next;
    }
    os << " }";
  }
    break;
  
  case FS_FS: {

    os << "[ ";
    indent += 2;

    int attr = a + core[a].next;

    if (attr != a) {
      os << core[attr].str << " = ";
      int nindent = indent + core[attr].str.size() + 3;
      prettyprint(attr + core[attr].offset, count, tag, nindent, os);
      attr += core[attr].next;
    }

    while (attr != a) {
      os << '\n';
      for (int i = 0; i < indent; ++i) { os << ' '; }
 
      os << core[attr].str << " = ";
      int nindent = indent + core[attr].str.size() + 3;
      prettyprint(attr + core[attr].offset, count, tag, nindent, os);
      attr += core[attr].next;
    }
    os << " ]";
  }
    break;

  case FS_VAL:
    os << core[a].feat;
    break;

  case FS_STRING:
    os << '"' << core[a].str << '"';
    break;

  case FS_CONSTR:
    os << "C\"" << core[a].str << '"';
    break;

  case FS_NOEXIST:
    os << "~EXIST";
    break;

  case FS_EXIST:
    os << "EXIST";
    break;

  case FS_UNSET:
    os << "UNSET";
    break;
  
  default:
    throw logic_error("fs::pprint: invalide node type : " + lexical_cast<string>(core[a].type));
    break;
  }
}

void featstruct::dump(int a, int & count, int * tag, ostream & os) const {

  a = follow(a);

  if (tag[a] < 0) { // node already passed, print reference
    os << '#' << -tag[a];
    return;
  }

  if (tag[a] > 1) { // first pass of a multipass node, print identifier
    os << "(" << count << ") ";
    tag[a] = -count;
    count++;
  }

  switch (core[a].type) {

  case FS_SET: {
    os << "{ ";
    int elem = a + core[a].next;
    while (elem != a) {
      dump(elem + core[elem].offset, count, tag, os);
      os << ", ";
      elem += core[elem].next;
    }
    os << "}";
  }
    break;

  case FS_FS: {
    os << "[ ";
    int attr = a + core[a].next;
    while (attr != a) {
      os << core[attr].str << " = ";
      dump(attr + core[attr].offset, count, tag, os);
      os << ", ";
      attr += core[attr].next;
    }
    os << "]";
  }
    break;

  case FS_VAL:
    os << core[a].feat;
    break;

  case FS_STRING:
    os << '"' << core[a].str << '"';
    break;

  case FS_CONSTR:
    os << "C\"" << core[a].str << '"';
    break;

  case FS_NOEXIST:
    os << "~EXIST";
    break;

  case FS_EXIST:
    os << "EXIST";
    break;

  case FS_UNSET:
    os << "UNSET";
    break;

  default:
    throw logic_error("fs::dump: bad node type : " + lexical_cast<string>(core[a].type));
    break;
  }
}


void featstruct::make_tag(int pos, int * tab) const {

  pos = follow(pos);

  if (tab[pos] == 0) { // first pass

    tab[pos] = 1;

    if (core[pos].type == FS_SET) {

      int elem = pos + core[pos].next;
      while (elem != pos) {
        make_tag(elem + core[elem].offset, tab);
        elem += core[elem].next;
      }

    } else if (core[pos].type == FS_FS) {

      int attr = pos + core[pos].next;
      while (attr != pos) {
        make_tag(attr + core[attr].offset, tab);
        attr += core[attr].next;
      }
    }

  } else { // second pass
    tab[pos] = 2;
  }
}


void featstruct::make_tag(int * tag) const {
  make_tag(entry, tag);
}

void featstruct::dump(std::ostream & os) const {

  if (! *this) { os << "[bottom]"; return; }

  int tag[size()];
  memset(tag, 0, size() * sizeof(int));
  make_tag(tag);
  int count = 1;
  dump(entry, count, tag, os);
}

void featstruct::prettyprint(ostream & os) const {

  if (! *this) { os << "[bottom]\n"; return; }

  int tag[size()];
  memset(tag, 0, size() * sizeof(int));
  make_tag(tag);
  int count = 1;
  prettyprint(entry, count, tag, 0, os);
  os << '\n';
}

featstruct::node * featstruct::get_entry_node() {
  if (! *this) { return 0; }
  return & core[entry];  
}

const featstruct::node * featstruct::get_entry_node() const {
  if (! *this) { return 0; }
  return & core[entry];  
}

#if 0
this is slower
bool featstruct::operator==(const featstruct & b) const {
  return equal2(get_entry_node(), b.get_entry_node());
}
#endif

bool featstruct::operator==(const featstruct & b) const {
  int sz = size() + b.size();
  int tags[sz];
  memset(tags, 0, sz * sizeof(int));
  return equal(get_entry_node(), b.get_entry_node(), tags + entry, tags + size() + b.entry);
}

