#include <iostream>
#include <set>
#include <vector>

#include <outilex/sentence_fsa.h>
#include <outilex/ugrammar.h>
#include <outilex/uchart.h>
#include <outilex/unification.h>
#include <outilex/earley_parser.h>
#include <outilex/featstruct.h>
#include <outilex/fs_node.h>


using namespace std;
using namespace boost;

/*
typedef boost::multi_index_container<
  e_item,
  indexed_by<
    sequenced<tag<by_pos> >,
    ordered_unique<tag<by_val>, identity<e_item> >
  >
> earley_stack;
*/

void earley_parser::e_item::dump(ostream & os) const {
  os << "{ (" << pfst->get_name() << ", " << q << "), [" << from << ", " << to << "], "
    << path  << ", w=" << w <<  "}";
}


namespace {

void check_item_path(const earley_parser::e_item & item, const uchart & chart) {

  const synt_path_type & path = item.path;
  for (synt_path_type::const_iterator it = path.begin(); it != path.end(); ++it) {
    int q = it->qno, transno = it->transno;
    if (q < 0 || q > chart.size()) {
      cerr << "bad item : " << item << endl;
      throw logic_error("internal earley error");
    }
    
    if (transno < 0) {
      transno = -transno - 1;
      if (transno > chart[q].size()) {
        cerr << "bad item: " << item << ": synttransno out of bound\n";
        throw logic_error("internal earley error");
      }
    } else {
      if (transno > chart.fsa.trans_size(q)) {
        cerr << "bad item: " << item << ": lextransno out of bound\n";
        throw logic_error("internal earley error");
      }
    }
  }
}

bool seems_like(const earley_parser::e_item & i1, const earley_parser::e_item & i2) {
  return i1.pfst == i2.pfst && i1.q == i2.q && i1.from == i2.from
    && i1.to == i2.to && i1.path == i2.path;
}

} // anonymous namespace

void earley_parser::ENQUEUE(int pos, const e_item & item) {

  //cerr << "ENQUEUE(" << pos << ", " << item  << ")\n";

  //  cerr << "ENQUEUE : item fs : " << item.fs << endl;

  agenda[pos].add(item);
}

void earley_parser::earley_stack::add(const e_item & item) {

  by_val_iterator where = item_set.lower_bound(item), end = by_val_end();
 
  for (by_val_iterator it = where; it != end && seems_like(item, *it); ++it) {
    if (it->fs == item.fs) { // same item keep best weight
      if (it->w < item.w) { // probably will never happen ?
        cerr << "earley stack: weighted optimisation :)\n";
        it->w = item.w;
      }
      return;
    }
  }

  by_val_iterator res = item_set.insert(where, item);
  items.push_back(res);
}


void earley_parser::parse(uchart & chart, bool surf) {

  //cerr << "earley parse\n";
 
  /* initialization stuffs */

  pchart = & chart;
  const sentence_fsa & txt = chart.fsa;
  
  agenda.clear();
  agenda.resize(txt.size());

  //cerr << "txt.size = " << txt.size() << endl;

  if (surf) {
    const string & axiom = gram.start_name();
    for (int pos = 0; pos < txt.size(); ++pos) {
      PREDICTOR(pos, axiom);
    }
  } else {
    PREDICTOR(0, gram.start_name());
  }

  for (int pos = 0; pos < txt.size(); ++pos) {

    //cerr << "POS = " << pos << endl;

    earley_stack & stack = agenda[pos];

    for (int i = 0; i < stack.size(); ++i) {
    
      //cerr << "\n" << pos << "/" << txt.size() << ":item #" << i <<  "/" << stack.size() << '\n';
      //cerr << "item = " << stack[i] << endl;
 
      //check_item(stack[i], chart);

      const usyntagm_pattern & fst = *(stack[i].pfst);
      int q = stack[i].q;

      //cerr << "path = " << stack[i].path << endl;

      if (q == -1) { // TERMINAL
        //cerr << "FAKE terminal\n";
        COMPLETER(pos, stack[i]);
        continue;
      }

      if (fst.final(q)) { // item is completed, add a completed itm (q == -1)
        //cerr << "COMPLETE, create a FAKE terminal item\n";
        const e_item & item = stack[i];
        ENQUEUE(pos, e_item(fst, -1, item.from, item.to, item.w, item.fs, item.path));
      } 


      for (usyntagm_pattern::const_trans_iterator str = fst.trans_begin(q), end = fst.trans_end(q);
           str != end; ++str) {

        if (str->in().type == input_type::SUBCALL) {

          const string & syntname = boost::get<string>(str->in().v);
 
          //cerr << "-> SUBCALL: " << syntname << endl;

          PREDICTOR(pos, syntname);
 
        } else if (str->in().type == input_type::LEXMASK) {

          const lexical_mask & m = boost::get<lexical_mask>(str->in().v);

          //cerr << "-> LEX: " << m << endl;

          for (int transno = 0; transno < txt.trans_size(pos); ++transno) {
 
            const sentence_fsa::transition tr = txt.get_trans(pos, transno); 
            const lexical_mask & e = tr.in();

            if (e.intersect(m)) {

              //cerr << tr.in() << " match with " << m << endl;

              const e_item & item = stack[i];

              synt_path_type npath(item.path);
              npath.push_back(syntref(pos, transno));

              if (! str->out().constraints.empty()) {

                featstruct fs(item.fs);
                fs.set("$$", tr.in());

                check_constraints(str->out().constraints, fs);

                if (fs) {
                  fs.unlink(fs_path(), "$$");
                  ENQUEUE(tr.to(), e_item(fst, str->to(), item.from, tr.to(),
                                          item.w + str->out().w, fs, npath));
                }
              } else {
                ENQUEUE(tr.to(), e_item(fst, str->to(), item.from, tr.to(),
                                        item.w + str->out().w, item.fs, npath));
              }
            }
          }

        } else {

          //cerr << "-> EPSILON\n";
 
          const e_item & item = stack[i];

          if (! str->out().constraints.empty()) {

            featstruct fs(item.fs);
            check_constraints(str->out().constraints, fs);

            if (fs) {
              ENQUEUE(pos, e_item(fst, str->to(), item.from, item.to, 
                                  item.w + str->out().w, fs, item.path));
            }

          } else {
            ENQUEUE(pos, e_item(fst, str->to(), item.from, item.to, 
                                item.w + str->out().w, item.fs, item.path));
          }
        }
      }
    }
    //cerr << "done with POS " << pos << " stack size = " << stack.size() << "\n\n";
  }
  //cerr << "out of early-parse\n";
}


void earley_parser::PREDICTOR(int pos, const std::string & syntname) {

  //cerr << "entering PREDICTOR(" << syntname << ")\n";

  int syntno = gram.get_syntagm_idx(syntname);
  //cerr << "syntno  = " << syntno << endl;

  if (syntno == -1) {
    cerr << "warning: unknow syntagm pattern : " << syntname << endl;
    return;
  }

  featstruct fs;
  fs.set("CAT", syntname);
  ENQUEUE(pos, e_item(gram[syntno], 0, pos, pos, 0, fs, synt_path_type()));
}



void earley_parser::COMPLETER(int pos, const e_item & item1) {

  uchart & chart = *pchart;

  const string & syntname = item1.pfst->get_name();

//  cerr << "COMPLETER(" << syntname << ")\n";
//  cerr << "fs = " << item1.fs << endl;

  //check_item(item1, chart);

  if (item1.from == item1.to) {
    cerr << "error: synt desc: " << syntname << " matches empty word\n";
    return;
  }
  
  int syntidx = chart.add_match(item1.from, syntagm(syntname, item1.to, item1.fs,
                                                    item1.path, item1.w));

  const featstruct & matchfs = chart.get_synt(item1.from, syntidx).fs;
  //cerr << "matchfs = " << matchfs << endl;

  syntref transref(item1.from, - syntidx - 1);
  double transw = item1.w;

  int curr = item1.to;

  earley_stack & stack = agenda[item1.from];

  // warning: do not use item1 ref in the following block,
  // because its adress can change .... (call to ENQUEUE)
  for (earley_stack::iterator it = stack.begin(), end = stack.end(); it != end; ++it) {
 
    const e_item & item2 = *it;
    const usyntagm_pattern & fst = *item2.pfst;
    const featstruct & curfs = item2.fs;
    int q = item2.q;

    if (q == -1) { continue; }

    for (usyntagm_pattern::const_trans_iterator str = fst.trans_begin(q),
         trans_end = fst.trans_end(q);
         str != trans_end; ++str) {

      if (str->in().type == input_type::SUBCALL
          && boost::get<string>(str->in().v) == syntname) {

        synt_path_type npath(item2.path);
        npath.push_back(transref);

        if (! str->out().constraints.empty()) {

          featstruct fs(curfs);
          fs.set("$$", matchfs);

          check_constraints(str->out().constraints, fs);
          if (fs) {
            fs.unlink(fs_path(), "$$");
            ENQUEUE(curr, e_item(fst, str->to(), item2.from, curr, item2.w + transw, fs, npath));
          }
        } else {
          ENQUEUE(curr, e_item(fst, str->to(), item2.from, curr, item2.w + transw, curfs, npath));
        }
      }
    }
  }
  //cerr << "out of COMPLETER\n";
}

