#include <iostream>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/lexical_cast.hpp>

#include <cstdlib>

#include <outilex/lingdef.h>

#include <outilex/ugrammar.h>
#include <outilex/xml_text_uchart.h>
#include <outilex/uchart.h>
#include <outilex/featstruct.h>
#include <outilex/fs_node.h>


using namespace std;
using namespace boost;

namespace fs = boost::filesystem;


namespace {

char * progname;

void usage() {
  cout << "usage: " << progname << " -l <lingdef> [-n <sentenceno>] [-form|-pos|-tags] [-w][-nofs] [-init] -synt <syntname> <chart>\n";
  exit(0);
}


enum anonymous {
  SHOW_FORM = 0, SHOW_POS, SHOW_ALL
} OUTPUT_MODE = SHOW_FORM;


const int MAX_FSLINE = 60;

bool SHOW_WEIGHT = false;
bool DUMP_FS = true;
bool INIT_STATE = false;

} // anonymous namespace


void dump_tree(const syntagm & synt, const uchart & chart, ostream & os) {

  os << "[." << synt.name << ' ';

  const vector<syntref> & path = synt.path;

  for (int i = 0; i < path.size(); ++i) {

    if (path[i].transno < 0) { // syntagm path

      dump_tree(chart.get_synt(path[i].qno, path[i].transno), chart, os);

    } else { // lexical entry

      os << '{';

      switch (OUTPUT_MODE) {

      case SHOW_FORM:
        os << chart.get_lex(path[i].qno, path[i].transno).in().form;  
        break;

      case SHOW_POS:
        os << chart.get_lex(path[i].qno, path[i].transno).in().form  
          << '.' << chart.get_lex(path[i].qno, path[i].transno).in().pos->get_name();
        break;

      case SHOW_ALL:
        os << chart.get_lex(path[i].qno, path[i].transno).in();  
        break;
      }
      os << '}';
    }
    os << ' ';
  }
  os << ']';

  //if (SHOW_WEIGHT) { os << "/" << synt.w; }
}


void LaTeX_dump_fs(const fs_node * n, int & count,
                   int & nbline, vector<string> & path,
                   int * tag, ostream & os) {

  if (nbline > MAX_FSLINE) { // break too large FS

    cerr << "break FS\n";

    int height = path.size();
    for (int i = 0; i < height; ++i) {
      os << "}";  
    }

    os << "\n\n";

    for (int i = 0; i < height; ++i) {
      os << "\\fs{." << path[i] << ": ";  
    }
    nbline = 0;
  }

  const fs_node * n2 = follow(n);
  tag += (n2 - n);
  n = n2;

  if (*tag < 0) { // node alread process
    os << "[" << -*tag << "] ";
    return;
  }

  if (*tag > 1) { // first pass of a multipass node
    os << "(" << count << ") ";
    *tag = -count;
    count++;
  }



  switch (n->type) {
 
  case FS_SET: {
    const fs_node * elem = n + n->next;
    while (elem != n) {
      LaTeX_dump_fs(elem + elem->offset, count,
                    nbline, path, tag + (elem + elem->offset - n), os);
      os << ", ";
      elem += elem->next;
    }
    os << ".";
  }
    break;

  case FS_FS: {
  
    os << "\\fs{";

    const fs_node * attr = n + n->next;

    while (attr != n) {
      os << attr->str  << ": "; //<< "[" << nbline << "]" << ": ";
      path.push_back(attr->str);
      const fs_node * v = attr + attr->offset;
      LaTeX_dump_fs(v, count, nbline, path, tag + (v - n), os);
      path.pop_back();
      os << "\\\\\n";
      ++nbline;
      attr += attr->next;
    }
    os << "} "; 
  }
    break;

  case FS_VAL:
    os << n->feat << " ";
    break;
  
  case FS_STRING:
    os << "\"" << n->str << "\" ";
    break;
  
  case FS_CONSTR:
    os << "C\"" << n->str << "\" ";
    break;
  
  case FS_NOEXIST:
    os << "!EXIST ";
    break;

  case FS_EXIST:
    os << "EXIST ";
    break;
  
  case FS_UNSET:
    os << "UNSET";
    break;
  
  default:
    throw logic_error("fs: dump_text: bad node type : " + lexical_cast<string>(n->type));
    break;
  }
}

void LaTeX_dump_fs(const featstruct & fs, ostream & os) {
  int tags[fs.size()];
  memset(tags, 0, fs.size() * sizeof(int));
  fs.make_tag(tags);
  int count = 0;
  int nbline = 0;
  vector<string> path;
#if 0
  os <<
    //"\\begin{figure}[ht]\n"
    "\\begin{center}\n";
#endif

  LaTeX_dump_fs(fs.get_entry_node(), count, nbline, path, tags + fs.entry, os);

#if 0
  os <<
    "\\end{center}\n"
    //"\\end{figure}\n"
    ;
#endif
}


int dump_synts(const uchart & chart, const string & syntname, ostream & os) {

  int num = 0;

  for (int q = 0; q < chart.size(); ++q) {

    uchart::const_by_name_iterator begin, end;
    chart.find(q, syntname, begin, end);

    while (begin != end) {
      //os << "match:\n";
      os << "\\subsection{match " << num << "}\n";
      const syntagm & synt = chart.get_synt(q, begin->second);
      os << "\\Tree ";
      dump_tree(synt, chart, os);
      os << '\n';
      if (DUMP_FS) { 
        LaTeX_dump_fs(synt.fs, os);
      }
      ++num;
      ++begin;
      os << '\n';
    }

    if (INIT_STATE) { // print synts only from initial state
      break;
    }
  }

  return num;
}


void latex_prolog(ostream & os) {

  os <<
    "\\documentclass[6pt,titlepage]{article}\n"
    "\n"
    //"\\usepackage{ucs}\n"
    //"\\usepackage[utf8]{inputenc}\n"
    "\\usepackage{qtree}\n"
    "\\usepackage{covington}\n"
    "\n"
    "\\begin{document}\n";
}


void latex_epilog(ostream & os) {
  os << "\\end{document}\n";
}



int main(int argc, char ** argv) try {

  fs::path chartpath, lingdefpath, texpath;
  string syntname = "mainP";
  int sentenceno = -1;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }


  progname = *argv;
  argv++, argc--;

  if (argc == 0) { usage(); }

  while (argc) {
    
    string arg = *argv;
 
    if ((arg == "-h") || (arg == "-help")) {
    
      usage();
    
    } else if (arg == "-l") {
    
      argv++, argc--;
      if (! argc) { cerr << "bad args: '-l' needs an argument\n"; exit(1); }
      lingdefpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-o") {
    
      argv++, argc--;
      if (! argc) { cerr << "bad args: '-o' needs an argument\n"; exit(1); }
      texpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-n") {

      argv++, argc--;
      if (! argc) { cerr << "bad args: '-n' needs an argument\n"; exit(1); }
      sentenceno = lexical_cast<int>(*argv);

    } else if (arg == "-synt") {

      argv++, argc--;
      if (! argc) { cerr << "bad args: '-synt' needs an argument\n"; exit(1); }
      syntname = *argv;

    } else if (arg == "-init") {
      
      INIT_STATE = true;
    
    } else if (arg == "-nofs") {
    
      DUMP_FS = false;

    } else if (arg == "-fs") {
    
      SHOW_WEIGHT = true;

    } else if (arg == "-form") {
    
      OUTPUT_MODE = SHOW_FORM;

    } else if (arg == "-pos") {
    
      OUTPUT_MODE = SHOW_POS;

    } else if (arg == "-tags") {
    
      OUTPUT_MODE = SHOW_ALL;

    } else {
      chartpath = fs::path(arg, fs::native);
    }
    argv++, argc--;
  }

  if (lingdefpath.empty() || chartpath.empty() || syntname.empty()) {
    cerr << progname << ": arguments missing\n";
    exit(1);
  }

  
  if (texpath.empty()) {
  
    string fname = chartpath.leaf();
    string::size_type dot = fname.rfind('.');

    if (dot != string::npos && fname.substr(dot) == ".gz") {
      fname.erase(dot);
      dot = fname.rfind('.');
    }

    if (sentenceno != -1) { fname += "." + lexical_cast<string>(sentenceno); }
    fname += ".tex";
    
    texpath = chartpath.branch_path() / fname;
  }


  ling_def lingdef(lingdefpath);
  unification_init(& lingdef);

  xml_itext_uchart ichart(chartpath, & lingdef);

  fs::ofstream out(texpath);

  latex_prolog(out);

  uchart chart;

  int nbmatch = 0;

  if (sentenceno != -1) { // proceed only one sentence
  
    if (! ichart.seek(sentenceno)) {
      cerr << "unable to read sentence #" << sentenceno << endl;
      exit(1);
    }
  
    ichart >> chart;
  
    out << "\\section{sentence " << sentenceno << "}\n"
      << chart.fsa.text << "\n\n";

    nbmatch = dump_synts(chart, syntname, out);
  
  } else {

    sentenceno = 0;
    while (ichart >> chart) {

    out << "\\section{sentence " << sentenceno << "}\n"
      << chart.fsa.text << "\n\n";

      nbmatch += dump_synts(chart, syntname, out);
      sentenceno++;
    }
  }

  latex_epilog(out);
  //cout << nbmatch << " matches.\n";

  return 0;

} catch (exception & e) {

  cerr << "fatal error: " << e.what() << endl;
  exit(1);

} catch (...) { cerr << "ouch!\n"; exit(1); }


