#include <iostream>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/lexical_cast.hpp>

#include <cstdlib>

#include <outilex/lingdef.h>
#include <outilex/text_fsa.h>

#include <outilex/ugrammar.h>
#include <outilex/xml_text_uchart.h>
#include <outilex/uchart.h>


using namespace std;
using namespace boost;

namespace fs = boost::filesystem;


namespace {

char * progname;

void usage() {
  cout << "usage: " << progname
    << " -l <lingdef> [-n <sentenceno>][-form|-pos|-tags][-w][-fs][-prefix|-affix|-whole]"
    "[-synt <syntname>][-id s:q:id] <chart>\n";
  exit(0);
}


enum anonymous1 {
  SHOW_FORM = 0, SHOW_POS, SHOW_ALL
} OUTPUT_MODE = SHOW_FORM;

enum anonymous2 {
  AFFIX, PREFIX, WHOLE
} MATCH_TYPE = AFFIX;

bool SHOW_WEIGHT = false;
bool DUMP_FS = false;
//bool INIT_STATE = false;

} // namespace ""


void dump_path(const syntagm & synt, const uchart & chart, ostream & os) {

  os << "(" << synt.name << ' ';
  const vector<syntref> & path = synt.path;

  for (int i = 0; i < path.size(); ++i) {

    if (path[i].transno < 0) { // syntagm path

      dump_path(chart.get_synt(path[i].qno, path[i].transno), chart, os);

    } else { // lexical entry
      
      switch (OUTPUT_MODE) {

      case SHOW_FORM:
        os << chart.get_lex(path[i].qno, path[i].transno).in().form;  
        break;

      case SHOW_POS:
        os << chart.get_lex(path[i].qno, path[i].transno).in().form  
          << '.' << chart.get_lex(path[i].qno, path[i].transno).in().pos->get_name();
        break;

      case SHOW_ALL:
        os << chart.get_lex(path[i].qno, path[i].transno).in();  
        break;
      }
    }
    os << ' ';
  }
  os << synt.name << ")";

  if (SHOW_WEIGHT) { os << "/" << synt.w; }
}



void dump_synt(const uchart & chart, int qno, int syntno, ostream & os) {

  const syntagm & synt = chart.get_synt(qno, syntno);

  dump_path(synt, chart, os);
  os << '\n';
  if (DUMP_FS) { 
    synt.fs.prettyprint(os);
  }
  os << '\n';
}




int dump_synts(const uchart & chart, const string & syntname, ostream & os) {

  int num = 0;

  for (int q = 0; q < chart.size(); ++q) {

    uchart::const_by_name_iterator begin, end;
    chart.find(q, syntname, begin, end);

    while (begin != end) {

      const syntagm & synt = chart.get_synt(q, begin->second);

      if (MATCH_TYPE == WHOLE && ! chart.fsa.final(synt.to)) { ++begin; continue; }

      os << "match:\n";
      dump_path(synt, chart, os);
      os << '\n';
      if (DUMP_FS) { 
        synt.fs.prettyprint(os);
      }
      ++num;
      ++begin;
      os << '\n';
    }

    if (MATCH_TYPE == PREFIX || MATCH_TYPE == WHOLE) { // print synts only from initial state
      break;
    }
  }

  return num;
}


int main(int argc, char ** argv) try {

  fs::path chartpath, lingdefpath;
  string syntname = "mainP";
  int sentenceno = -1, qno = -1, syntno = -1;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }


  progname = *argv;
  argv++, argc--;

  if (argc == 0) { usage(); }

  while (argc) {
    
    string arg = *argv;
 
    if ((arg == "-h") || (arg == "-help")) {
    
      usage();
    
    } else if (arg == "-l") {
    
      argv++, argc--;
      if (! argc) { cerr << "bad args: '-l' needs an argument\n"; exit(1); }
      lingdefpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-n") {

      argv++, argc--;
      if (! argc) { cerr << "bad args: '-n' needs an argument\n"; exit(1); }
      sentenceno = lexical_cast<int>(*argv);

    } else if (arg == "-id") {

      argv++, argc--;
      if (! argc) { cerr << "bad args: '-id' needs an argument\n"; exit(1); }
      if (sscanf(*argv, "%d:%d:%d", & sentenceno, & qno, & syntno) != 3) {
        cerr << "bad argument for id\n";
        exit(1);
      }

    } else if (arg == "-synt") {

      argv++, argc--;
      if (! argc) { cerr << "bad args: '-synt' needs an argument\n"; exit(1); }
      syntname = *argv;

    } else if (arg == "-affix") {
      
      MATCH_TYPE = AFFIX;
    
    } else if (arg == "-prefix") {
      
      MATCH_TYPE = AFFIX;
    
    } else if (arg == "-whole") {
      
      MATCH_TYPE = WHOLE;
    
    } else if (arg == "-fs") {
    
      DUMP_FS = true;

    } else if (arg == "-w") {
    
      SHOW_WEIGHT = true;

    } else if (arg == "-form") {
    
      OUTPUT_MODE = SHOW_FORM;

    } else if (arg == "-pos") {
    
      OUTPUT_MODE = SHOW_POS;

    } else if (arg == "-tags") {
    
      OUTPUT_MODE = SHOW_ALL;

    } else {
      chartpath = fs::path(arg, fs::native);
    }
    argv++, argc--;
  }

  if (lingdefpath.empty() || chartpath.empty() || syntname.empty()) {
    cerr << progname << ": arguments missing\n";
    exit(1);
  }


  ling_def lingdef(lingdefpath);

  unification_init(& lingdef);

  xml_itext_uchart ichart(chartpath, & lingdef);

  uchart chart;

  int nbmatch = 0;

  if (syntno != -1) {
  
    if (! ichart.seek(sentenceno)) {
      cerr << "unable to read sentence #" << sentenceno << endl;
      exit(1);
    }

    ichart >> chart;

    if (chart.size() < qno) {
      cerr << "sentence " << sentenceno << " too small (size=" << qno << ")\n";
    }
    dump_synt(chart, qno, syntno, cout);

  } else if (sentenceno != -1) { // proceed only one sentence
  
    if (! ichart.seek(sentenceno)) {
      cerr << "unable to read sentence #" << sentenceno << endl;
      exit(1);
    }
  
    ichart >> chart;
  
    cout << "sentence #" << sentenceno << "\n"
      << chart.fsa.text << "\n\n";

    nbmatch = dump_synts(chart, syntname, cout);
  
    cout << nbmatch << " match(es).\n";

  } else {

    sentenceno = 0;
    while (ichart >> chart) {

      cout << "sentence #" << sentenceno << "\n"
        << chart.fsa.text << "\n\n";

      nbmatch += dump_synts(chart, syntname, cout);
      sentenceno++;
    }
    cout << nbmatch << " match(es).\n";
  }


  return 0;

} catch (exception & e) {

  cerr << "fatal error: " << e.what() << endl;
  exit(1);

} catch (...) { cerr << "ouch!\n"; exit(1); }


