#include <iostream>
#include <vector>

#include <outilex/sentence_fsa.h>
#include <outilex/syntref.h>
#include <outilex/lexical_entry.h>
#include <outilex/deter_string_fst.h>
#include <outilex/wgb_color.h>

#include <outilex/transduction2.h>

using namespace std;


namespace {

typedef deter_string_fst::output_type output_type;

struct str_match_res {

  int to;               // destination state of the matching path in text fsa 
  vector<syntref> path; // path of the "better" matching sequence : sequence of (qno, transno)
  output_type     out;  // output associated to path

  str_match_res() : to(-1), path(), out() {}
  str_match_res(int q, const output_type & o) : to(q), path(), out(o) {}
  str_match_res(int q, const syntref & ref, const output_type & o) : to(q), path(1, ref), out(o) {}
  str_match_res(int q, const vector<syntref> & p, const output_type & o) : to(q), path(p), out(o) {}


  bool operator<(const str_match_res & b) const {
  
    if (to != b.to) { // best path is the one matching the longest segment in text
      return to < b.to;
    }

    // if segment are the same, best path is the one with better output

    if (out < b.out) {
      return true;
    } else if (b.out < out) {
      return false;
    }

    // if segment and output are equivalent,
    // best path is one passing with compounds... else no best path
 
    return path.size() > b.path.size();
  }

  // store the best matching path between *this and b parameter
  str_match_res & operator+=(const str_match_res & b) {
    if (*this < b) {
      *this = b;
    }
    return *this;
  }

  // matching path concatenation
  str_match_res & operator*=(const str_match_res & b) {
    to = b.to;
    path.insert(path.end(), b.path.begin(), b.path.end());
    out = out.delay(b.path.size());
    out *= b.out;
    return *this;
  }

  // matching path concatenation
  static str_match_res mult(const str_match_res & a, const str_match_res & b) {
    str_match_res res(b.to, a.path, a.out.delay(b.path.size())); // we delay out by b.path.size()
    // concatenate path
    res.path.insert(res.path.end(), b.path.begin(), b.path.end()); 
    // merge output
    res.out *= b.out;
    return res;
  }

};




struct cache_fst {

  typedef map<pair<int, int>, int> states_map;

  struct state {

    wgb_color color;
    str_match_res res;

    state() : color(white), res() {}
  };

  sentence_fsa & text;
  deter_string_fst & T;

  vector<state> states;
  states_map    states_id;

  cache_fst(sentence_fsa & fsa, deter_string_fst & T_) : text(fsa), T(T_), states(), states_id() {}

  inline wgb_color & color(int q) { return states[q].color; }
  inline str_match_res & res(int q) { return states[q].res; }

  int add_state(int q1, int q2) {
  
    pair<int,int> id(q1, q2);
    if (states_id.find(id) != states_id.end()) { return states_id[id]; }

    int res = states.size();
    states.resize(res + 1);

    states_id[id] = res;

    return res;
  }

  int inter_state(int q1, int q2);
};



int cache_fst::inter_state(int q1, int q2) {

  //cerr << "entering inter_state: (" << q1 << ", " << q2 << ")\n";

  int q = add_state(q1, q2);

  if(color(q) == black) { // we've already been there
//    cerr << "inter_state: already computed!\n";
    return q;  
  }

  assert(color(q) == white); // assert this is the first time we reach this state

  color(q) = gray;


  if (T.final(q2)) { // we reach a final state in T : we found at least one matching sequence

#warning we can assume that final outputs are already sorted (in std::set) should take the last one

    //cerr << "inter_state: " << q2 << " is final!\n";
    for (deter_string_fst::outputs_type::iterator it = T.final_outputs(q2).begin();
         it != T.final_outputs(q2).end(); ++it) {

      res(q) += str_match_res(q1, *it);
    }
  }


  // iterate on transition in text leaving q1
  for (int transno = 0; transno < text.trans_size(q1); ++transno) {
    
    const sentence_fsa::transition & tr = text.get_trans(q1, transno);
    
 //   cerr << "interstate : looking for " << tr.in() << endl;

    deter_string_fst::const_trans_iterator gtr = T.find_matching_trans(q2, tr.in());
  
    if (gtr != T.trans_end(q2) && gtr->to() != -1) { // we found a matching transition in T from q2

//      cerr << "found! label=" << gtr->in() << ", output=" << gtr->out() << endl;

      str_match_res tmpres(tr.to(), syntref(q1, transno), gtr->out());

      int to = inter_state(tr.to(), gtr->to());

//      cerr << "back to inter (" << q1 << ", " << q2 << ")\n";
      tmpres *= res(to);
      res(q) += tmpres;

    } //else { cerr << "no matching transition found.\n"; }
  }

  color(q) = black;
  return q;
}

}; // namespace ""


void output_res(const sentence_fsa & text, str_match_res & res, ostream & os, int transmode,
                int outputmode) {

//  cerr << "output_res...\n";


  if (outputmode == HTML_OUTPUT) {
    os << "<strong>";
  }
  
  int size = res.path.size();
 // cerr << "size = " << size << endl;
  vector<string> outputs(size);

  assert(res.out != output_type::zero());

  if (transmode == IGNORE_MODE) {

    for (int i = 0; i < size; ++i) {
      syntref & ref = res.path[i];
      os << text.get_trans(ref.qno, ref.transno).in().form << ' ';
    }

  } else { // soit MERGE soit REPLACE

    for (map<int, string>::const_iterator it = res.out.v.begin(), end = res.out.v.end();
         it != end; ++it) {
      assert(it->first < size);
      outputs[size - 1 - it->first] = it->second;
    }

    if (transmode == REPLACE_MODE) {
    
      for (int i = 0; i < size; ++i) {
        if (! outputs[i].empty()) { os << outputs[i] << ' '; }
      }

    } else { // MERGE_MODE

      for (int i = 0; i < size; ++i) {
        if (! outputs[i].empty()) { os << outputs[i] << ' '; }
        syntref & ref = res.path[i];
        os << text.get_trans(ref.qno, ref.transno).in().form << ' ';
      }
    }
  }


//  cerr << "bye\n";
  if (outputmode == HTML_OUTPUT) {
    os << "</strong>";
  }
}


void output_trans(const sentence_fsa::transition & tr, ostream & os, int transmode,
                  int outputmode) {
  os << tr.in().form << ' ';
}


//void transduction(sentence_fsa & text, deter_string_fst & T, std::ostream & os) {
int transduction2(sentence_fsa & text, deter_string_fst & T, std::ostream & os,
                  int transmode, int outputmode) {

  int nbmatches = 0;

  cache_fst cache(text, T);
  
  int q = 0;
  while (! text.final(q)) { // stop when we reach the final state
    
//    cerr << "q = " << q << endl;

    int idx = cache.inter_state(q, T.start());

    str_match_res & res = cache.res(idx);

    if (res.to != -1) { // we find (at least) one matching sequence

//      cerr << "find a match\n";

      ++nbmatches;
      output_res(text, res, os, transmode, outputmode);
      q = res.to;

    } else { // we output first transition

//      cerr << "no match\n";

      if (text.trans_size(q) == 0) {
        os << "*ERROR*\n";
        cerr << "ERROR: bad text automaton\n";
        return nbmatches;
      }
      assert(text.trans_size(q) > 0);
      output_trans(text.get_trans(q, 0), os, transmode, outputmode); 
      q = text.get_trans(q,0).to();
    }
  }

 // cerr << "out of loop\n";

  if (q < text.size() - 1) { cerr << "error: sentence fsa not topologiacly sorted\n"; }

  if (outputmode == HTML_OUTPUT) {
    os << "\n<br/>\n";
  } else {
    os << '\n';
  }
  return nbmatches;
}

