#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>

#include <outilex/lingdef.h>
#include <outilex/string_transducer.h>
#include <outilex/fsa-prune.h>
#include <outilex/epsilon_removal.h>


using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

namespace {

char * progname;

void usage() {
  cout << "usage : " << progname << " [-r|-p] [-o <sfst> ] <sfst>\n";
  exit(1);
}

struct is_epsilon {
  pos_def * pos_epsilon;
  
  is_epsilon(pos_def * pos) : pos_epsilon(pos) {}
  bool operator()(const lexical_mask & m) const { return m.get_pos() == pos_epsilon; }
};


} // namespace ""

int main(int argc, char ** argv) try {

  fs::path ipath, opath, lingdefpath;
  bool epsilon_cleanup = false;
  bool prune_cleanup = false;


  {
    char * text = getenv("LINGDEF");
    if (text) {
      lingdefpath = fs::path(text, fs::native);  
    }
  }


  progname = *argv;
  argv++, argc--;

  if (! argc) { usage(); }

  while (argc) {
  
    string arg = *argv;

    if (arg == "-h") {
    
      usage();
    
    } else if (arg == "-p") {

      prune_cleanup = true;

    } else if (arg == "-r") {

      epsilon_cleanup = true;

    } else if (arg == "-l") {
    
      argv++, argc--;
      if (! argc) { cerr << "bad args\n"; exit(1); }

      lingdefpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-o") {
    
      argv++, argc--;
      if (! argc) { cerr << "bad args\n"; exit(1); }

      opath = fs::path(*argv, fs::native);
    
    } else {
      ipath = fs::path(*argv, fs::native);
    }

    argv++, argc--;
  }

  if (ipath.empty() || lingdefpath.empty()) {
    cerr << "arg missing\n";
    exit(1);
  }

  if (opath.empty()) { opath = ipath; }

  ling_def lingdef(lingdefpath);

  cerr << "loading " << ipath.string() << endl;

  string_transducer fst(ipath, & lingdef);

  if (prune_cleanup) {
    cerr << "pruning ...\n";
    fst_prune(fst);
  }

  if (epsilon_cleanup) {
    cerr << "epsilon_removal ...\n";
    fst_remove_epsilon(fst, is_epsilon(lingdef.epsilon_pos()));
    cerr << "(re-)pruning...\n";
    fst_prune(fst);
  }

  cerr << "done.\n";

  fst.write(opath);

} catch (exception & e) {
  cerr << "fatal error: " << e.what() << endl;
  exit(1);
}

