#include <iostream>
#include <stack>

#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/convenience.hpp>
                       
#include <outilex/FST2_to_string_transducer.h>
#include <outilex/string_transducer.h>

#include <outilex/epsilon_removal.h>
#include <outilex/lingdef.h>

using namespace std;

namespace fs = boost::filesystem;


char * progname;


void usage(ostream & os = cerr, int ret = 1) {
  os << "usage: " << progname << " [-r] <fst2>\n";
  exit(ret);
}

struct is_epsilon {
  pos_def * pos_epsilon;
  
  is_epsilon(pos_def * pos) : pos_epsilon(pos) {}
  bool operator()(const lexical_mask & m) const { return m.get_pos() == pos_epsilon; }
};


int main(int argc, char ** argv) try {

  fs::path lingdefpath, fstpath;
  bool remove_epsilon = false;


  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }


  progname = *argv;
  argv++, argc--;

  if (argc == 0) { usage(); }


  while (argc) {

    string arg = *argv;

    if (arg == "-h") {

      usage(cout, 0);

    } else if (arg == "-r") {

      remove_epsilon = true;

    } else if (arg == "-l") {

      argv++, argc--;
      if (argc == 0) { usage(); }
      lingdefpath = fs::path(*argv, fs::native);

    } else { // assume fst name

      fstpath = fs::path(*argv, fs::native);
    }
  
    argv++, argc--;
  }

  if (lingdefpath.empty() || fstpath.empty()) {
    cerr << "error: argument missing\n";
    exit(1);
  }


  ling_def lingdef(lingdefpath);


  fs::path outpath = fs::change_extension(fstpath, ".sfst");

  cout << "compiling " << fstpath.string() << " into " << outpath.string() << "...\n";


  string_transducer T(& lingdef);

  FST2_to_string_transducer(fstpath, T);

  T.check_for_final_outputs();

  if (remove_epsilon) {
    cout << "epsilon removal ...\n";
    fst_remove_epsilon(T, is_epsilon(lingdef.epsilon_pos()));
  }
  T.write(outpath);
  cout << "done. result in " << outpath.string() << ".\n";

} catch (exception & e) {
  cerr << "fatal error: " << e.what() << endl;
  exit(1);
}

