#include <iostream>
#include <vector>
#include <map>
#include <string>
#include <stdexcept>

#include <boost/filesystem/path.hpp>
#include <outilex/text_fsa.h>

using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

namespace {

struct fsa_reader_item {
  ifsa_guesser_f guess;
  ifsa_creator_f create;
  fsa_reader_item(ifsa_guesser_f g, ifsa_creator_f c) : guess(g), create(c) {}
};

typedef std::vector<fsa_reader_item> readers_type;
readers_type * text_fsa_readers = 0;

readers_type & get_readers() {
  if (text_fsa_readers == 0) { text_fsa_readers = new readers_type(); } // memory leak !
  return *text_fsa_readers;
}


typedef map<string, ofsa_creator_f> writers_type;
writers_type * text_fsa_writers = 0;

writers_type & get_writers() {
  if (text_fsa_writers == 0) { text_fsa_writers = new writers_type(); } // memory leak !
  return *text_fsa_writers;
}

} // namespace ""


void register_text_fsa_reader(ifsa_guesser_f guess, ifsa_creator_f create) {
  readers_type & readers = get_readers();
  readers.push_back(fsa_reader_item(guess, create));
}


void register_text_fsa_writer(const string & format, ofsa_creator_f create) {
  writers_type & writers = get_writers();
  writers[format] = create;
}

itext_fsa * new_itext_fsa(const fs::path & path, ling_def * lingdef) {
  readers_type & readers = get_readers();
  for (int i = 0; i < readers.size(); ++i) {
    if (readers[i].guess(path)) { return readers[i].create(path, lingdef); }
  }
  throw runtime_error("new_itext_fsa: invalid format for file : " + path.string());
}

bool is_text_fsa(const fs::path & path) {
  readers_type & readers = get_readers();
  for (int i = 0; i < readers.size(); ++i) {
    if (readers[i].guess(path)) { return true; }
  }
  return false;
}

otext_fsa * new_otext_fsa(const fs::path & path, const string & format, int size,  int compression) {
  writers_type & writers = get_writers();
  writers_type::iterator it = writers.find(format);
  if (it == writers.end()) { throw runtime_error("new_otext_fsa: invalid format : " + format); }
  return it->second(path, size, compression);
}



// we have to put something like this in this compilation unit
// to resolve some link problems...

// this file will be linked (with the routines above) with program
// that make use of [io]_text_fsa

itext_fsa::~itext_fsa() {}
otext_fsa::~otext_fsa() {}


// we load text fsa formaters when this file is linked 

extern int LINK_WITH_XML_TEXT_FSA;
extern int LINK_WITH_BIN_TEXT_FSA;
extern int LINK_WITH_TXT_TEXT_FSA;

// never called
void __dumb_text_fsa_linker_helper__() {
  cerr << "dumb_linker_helper: who call me ???\n";
  LINK_WITH_XML_TEXT_FSA++;
  LINK_WITH_BIN_TEXT_FSA++;
  LINK_WITH_TXT_TEXT_FSA++;
}


