#include <sstream>

#include <boost/filesystem/fstream.hpp>
#include <boost/lexical_cast.hpp>
#include <outilex/xml.h>

#include <outilex/syntagm_pattern.h>

using namespace std;
using namespace boost;

namespace fs = boost::filesystem;

void syntagm_pattern::transition::write_XML(xmlwriter & writer) const {

  writer.start_element("transition");
  writer.write_attribute("to", lexical_cast<string>(to_));
  writer.start_element("in"); 
  in_.write_XML(writer);
  writer.end_element();
  writer.start_element("out"); 
  out_.write_XML(writer);
  writer.end_element();
  writer.end_element();
}


void syntagm_pattern::transition::read_XML(xmlNodePtr node, ling_def * lingdef) {

  char * text = (char *) xmlGetProp(node, (const xmlChar *) "to");
  istringstream(text) >> to_;
  xmlFree(text);

  bool okin = false, okout = false;
  node = node->xmlChildrenNode;

  while (node) {

    if (xmlStrcmp(node->name, (const xmlChar *) "in") == 0) {

      xmlNodePtr cur = node->xmlChildrenNode;

      while (cur) {
        if (xmlStrcmp(cur->name, input_type::xml_name()) == 0) {
          in_.read_XML(cur, lingdef);
          okin = true;
        }
        cur = cur->next;
      }

    } else if (xmlStrcmp(node->name, (const xmlChar *) "out") == 0) {

      xmlNodePtr cur = node->xmlChildrenNode;

      while (cur) {
        if (xmlStrcmp(cur->name, output_type::xml_name()) == 0) {
          out_.read_XML(cur);
          okout = true;
        }
        cur = cur->next;
      }
    }

    node = node->next;
  }

  if (!okin || !okout) {
    throw xml_parse_error("syntagm_pattern::read_XML: bad transition");
  }
}


void syntagm_pattern::synt_transition::write_XML(xmlwriter & writer) const {
  writer.start_element("synt_transition");
  writer.write_attribute("to", lexical_cast<string>(to_));
  writer.start_element("in"); 
  in_.write_XML(writer);
  writer.end_element();
  writer.start_element("out"); 
  out_.write_XML(writer);
  writer.end_element();
  writer.end_element();
}

void syntagm_pattern::synt_transition::read_XML(xmlNodePtr node, ling_def * lingdef) {

  char * text = (char *) xmlGetProp(node, (const xmlChar *) "to");
  istringstream(text) >> to_;
  xmlFree(text);

  node = node->xmlChildrenNode;
  while (node) {

    if (xmlStrcmp(node->name, (const xmlChar *) "in") == 0) {

      xmlNodePtr cur = node->xmlChildrenNode;

      while (cur) {
        if (xmlStrcmp(cur->name, synt_input_type::xml_name()) == 0) {
          in_.read_XML(cur, lingdef);
        }
        cur = cur->next;
      }

    } else if (xmlStrcmp(node->name, (const xmlChar *) "out") == 0) {

      xmlNodePtr cur = node->xmlChildrenNode;

      while (cur) {
        if (xmlStrcmp(cur->name, synt_output_type::xml_name()) == 0) {
          out_.read_XML(cur);
        }
        cur = cur->next;
      }
    }

    node = node->next;
  }
  //cerr << "synt_tran:read_XML: in=" << in_ << endl;
}


void syntagm_pattern::state::write_XML(xmlwriter & writer) const {

  writer.start_element("state");

  if (final) {
    writer.write_attribute("final", "1");
  }

  for (const_trans_iterator it = trans.begin(); it != trans.end(); ++it) {
    (*it).write_XML(writer);
  }

  for (const_synt_trans_iterator it = synt_trans.begin(); it != synt_trans.end(); ++it) {
    (*it).write_XML(writer);
  }
  writer.end_element();
}

void syntagm_pattern::state::read_XML(xmlNodePtr node, ling_def * lingdef) {

  clear();

  char * text = (char *) xmlGetProp(node, (xmlChar *) "final");

  if (text) {
    istringstream(text) >> final;
  } else { final = false; }

  node = node->xmlChildrenNode;

  while (node) {

    if (xmlStrcmp(node->name, (const xmlChar *) "transition") == 0) {

      int size = trans.size();

      try {

        trans.resize(size + 1);
        trans[size].read_XML(node, lingdef);

      } catch (xml_parse_error & e) {

        cerr << "unable to load transition: " << e.what() << '\n';
        trans.resize(size);
      }

    } else if (xmlStrcmp(node->name, (const xmlChar *) "synt_transition") == 0) {

      int size = synt_trans.size();

      try {

        synt_trans.resize(size + 1);
        synt_trans[size].read_XML(node, lingdef);

      } catch (xml_parse_error & e) {

        cerr << "unable to load synt transition: " << e.what() << '\n';
        synt_trans.resize(size);
      }
    }
    node = node->next;
  }
}

void syntagm_pattern::write_XML(xmlwriter & writer) const {

  writer.start_element("syntagm_pattern");
  writer.write_attribute("name", name);
  writer.write_attribute("size", lexical_cast<string>(states.size()));

  for_each(states.begin(), states.end(), XML_Writer(writer));

  writer.end_element();
}

void syntagm_pattern::write_XML(const fs::path & path) const {
  xmlwriter writer(path);
  writer.start_document();
  write_XML(writer);
  writer.end_document();
}

void syntagm_pattern::read_XML(xmlNodePtr node, ling_def * ldef) {

  //cerr << "syntagm_pattern::read_XML\n";

  lingdef = ldef;
  clear();

  char * text = xmlGetProp(node, "name");
  name = text;
  xmlFree(text);

  text = xmlGetProp(node, "size");
  if (text) {
    int size;
    istringstream(text) >> size;
    states.reserve(size);
    xmlFree(text);
  }

  node = node->xmlChildrenNode;

  int qno = 0;
  while (node) {
    if (xmlStrcmp(node->name, "state") == 0) {
      states.resize(qno + 1);
      states[qno].read_XML(node, lingdef);
      qno++;
    }
    node = node->next;
  }
  cerr << "pattern " << name << " loaded (" << size() << " states)\n";
}


void syntagm_pattern::read(const fs::path & path, ling_def * ldef) {

  lingdef = ldef;

  string fname(path.native_file_string());

  xmlDocPtr doc = xmlParseFile(fname.c_str());

  if (doc == NULL) {
    throw xml_parse_error("syntagm_pattern::read_XML: file '" + fname + "' not successfully parsed");
  }

  xmlNodePtr node = xmlDocGetRootElement(doc);

  try {
    read_XML(node, lingdef);
  } catch (exception & e) {
    cerr << "syntagm_pattern::read: error while parsing grammar '"
      << fname << "' : " << e.what() << endl;
    xmlFreeDoc(doc);
    throw e;
  }

  xmlFree(doc);
}
