#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <stdexcept>

#include <boost/filesystem/fstream.hpp>
#include <boost/lexical_cast.hpp>

#include <outilex/xml.h>

#include <outilex/grammar.h>

using namespace std;
using namespace boost;

namespace fs = boost::filesystem;


grammar::grammar(const fs::path & path_, ling_def * ldef)
  : syntagms(), synt_map() {
  read(path_, ldef);
}

void grammar::read(const fs::path & path_, ling_def * ldef) {

  lingdef = ldef;
  path    = path_;
  string fname   = path.native_file_string();

  xmlDocPtr doc = xmlParseFile(fname.c_str());

  if (doc == NULL ) {
    throw xml_parse_error("grammar::read_XML: file '" + fname + "' not successfully parsed");
  }

  xmlNodePtr node = xmlDocGetRootElement(doc);

  try {
    read_XML(node, lingdef);
  } catch (exception & e) {
    cerr << "grammar:read: error while parsing grammar '" << fname << "' : " << e.what() << endl;
    xmlFreeDoc(doc);
    throw;
  }

  xmlFreeDoc(doc);
}

void grammar::write(const fs::path & filename) const {
  xmlwriter writer(filename);
  writer.start_document();
  write_XML(writer);
  writer.end_document();
}



int grammar::add_pattern(const fs::path & patternpath, const std::string & name) {

  cerr << "grammar::add_pattern: " << patternpath.native_file_string() << " " << name << endl;

  map<string, int>::iterator it = synt_map.find(name);

  int idx;

  if (it == synt_map.end()) { // new syntagm
    
    idx = syntagms.size();
    syntagms.resize(idx + 1);

    synt_map[name] = idx;

  } else { // override
    idx = it->second;  
  }

  cerr << "idx=" << idx << endl;

  syntagms[idx].init(syntagm_pattern(patternpath, lingdef));
  cerr << "getname=" << syntagms[idx].get_name() << endl;
  assert(syntagms[idx].get_name() == name);

  return idx;
}


void grammar::write_XML(xmlwriter & writer) const {

  writer.start_element("grammar");
  writer.write_attribute("main", syntagms[start_graph].get_name());
  writer.write_attribute("size", lexical_cast<string>(syntagms.size()));

  for_each(syntagms.begin(), syntagms.end(), XML_Writer(writer));
  writer.end_element();
}

#if 0
void grammar::dump_XML(ostream & os) const {

  cerr << "gramm:dump_XML: start =" << start_graph << endl;

  os << "<grammar main='" << syntagms[start_graph].get_name()
    << "' size='" << syntagms.size() << "'>";
  for_each(syntagms.begin(), syntagms.end(), XML_Dumper(os));
  os << "</grammar>";
}
#endif

void grammar::read_XML(xmlNodePtr node, ling_def * ldef) {

  lingdef = ldef;

  syntagms.clear();
  synt_map.clear();

  if (node == NULL) {
    throw xml_parse_error("grammar::read_XML: '"
                          + path.native_file_string() + "': document is empty");
  }

  if (xmlStrcmp(node->name, "grammar")) {
    throw xml_parse_error("grammar::read_XML: '"
                          + path.native_file_string() + "': wrong document type");
  }

  char * text = xmlGetProp(node, "main");
  string mainname = text;
  xmlFree(text);

 // cerr << "main = " << mainname << endl;

  text = xmlGetProp(node, "size");
  if (text) {
    int size;
    istringstream(text) >> size;
    syntagms.reserve(size);
  }

  node = node->xmlChildrenNode;

  int syntno = 0;
  while (node) {

    if (xmlStrcmp(node->name, syntagm_fst::xml_name()) == 0) {

      cerr << "loading synt_fst\n";

      syntagms.resize(syntno + 1);
      syntagms[syntno].read_XML(node, lingdef);
      synt_map[syntagms[syntno].get_name()] = syntno;
      ++syntno;

    } else if (xmlStrcmp(node->name, syntagm_pattern::xml_name()) == 0) {

      cerr << "loading synt_pattern\n";

      syntagms.resize(syntno + 1);
      syntagms[syntno].read_XML(node, lingdef);
      synt_map[syntagms[syntno].get_name()] = syntno;
      ++syntno;
    }
    node = node->next;
  }

  if (! mainname.empty()) {
    start_graph = synt_map[mainname];
  } else { start_graph = 0; }
}


