#include <iostream>
#include <fstream>
#include <stdexcept>
#include <sstream>
#include <vector>

#include <boost/progress.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/fstream.hpp>

#include <outilex/text_fsa.h>
#include <outilex/sentence_fsa.h>
#include <outilex/rtn_grammar.h>
#include <outilex/rtn_parser.h>
#include <outilex/rtn_transduct.h>


namespace {

using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

char * progname;


void usage() {
  cerr << "usage: " << progname
    << " -l <lingdef> -rtn <fst> [ -html | -txt ] [ -concord | -trans ] [-m|-r|-i] [ -o <outputres> ] <txtfsa>\n";
  exit(1);
}

enum transmode_type { TRANSDUCTION, CONCORDANCE };

}; // namespace ""

int main(int argc, char ** argv) try {

  fs::path txtpath, lingdefpath, rtnpath, opath;

  char * text = getenv("LINGDEF");
  if (text) {
    lingdefpath = fs::path(text, fs::native);
  }

  trans_mode_type transmode  = MERGE;
  trans_output_type outputmode = TEXT;

  int MODE = TRANSDUCTION;

  progname = *argv;

  argv++, argc--;

  while (argc) {
    
    string arg = *argv;
    
    if (arg == "-l") {
      
      argv++, argc--;
      if (argc == 0) { usage(); }
      lingdefpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-o") {
      
      argv++, argc--;
      if (argc == 0) { usage(); }
      opath = fs::path(*argv, fs::native);
    
    } else if (arg == "-rtn") {
      
      argv++, argc--;
      if (argc == 0) { usage(); }
      rtnpath = fs::path(*argv, fs::native);
    
    } else if (arg == "-html") {
    
      outputmode = HTML;
    
    } else if (arg == "-text") {
    
      outputmode = TEXT;
    
    } else if (arg == "-concord") {
    
      MODE = CONCORDANCE;
    
    } else if (arg == "-trans") {
    
      MODE = TRANSDUCTION;
    
    } else if (arg == "-m") {
    
      transmode = MERGE;
    
    } else if (arg == "-r") {
    
      transmode = REPLACE;
    
    } else if (arg == "-i") {
    
      transmode = IGNORE;
    
    } else if (arg == "-h") {
    
      usage();
    
    } else {

      txtpath = fs::path(arg, fs::native);
    }

    argv++, argc--;
  }

  if (txtpath.empty() || lingdefpath.empty() || rtnpath.empty()) { usage(); }

  if (opath.empty()) {

    string resname;
    if (MODE == TRANSDUCTION) {
      resname = "transduction-res";
    } else {
      resname = "concord";
    }

    if (outputmode == HTML) {
      resname += ".html";
    } else {
      resname += ".txt";
    }
    opath = resname;
  }
  
  fs::ofstream os(opath);
  if (! os) { cerr << "unable to open " << opath.string() << endl; exit(1); }

  if (outputmode == HTML) {
    os << 
      "<html>\n"
      "<head>\n"
      "<meta http-equiv=\"Content-Type\" content=\"text/html;charset=utf-8\"/>\n"
      "<title>" << txtpath.leaf() << " : " <<
      ((MODE == TRANSDUCTION) ? "Transduction by " : "Matches with ")
      << rtnpath.leaf() << "</title>\n" <<
      "</head>\n"
      "<body bgcolor='#ffffff' text='#000000'>\n"
      "<h1>" << txtpath.leaf() << "</h1>\n";
  }


  ling_def lingdef(lingdefpath);

  scoped_ptr<itext_fsa> p_itext(new_itext_fsa(txtpath, & lingdef));
  itext_fsa & itext = *p_itext;

  int total = itext.size();

  rtn_grammar rtn(rtnpath, & lingdef);

  rtn_parser parser(rtn);

  sentence_fsa fsa;

  int sentenceno = 0, totalmatches = 0;
  progress_display show_progress(total, cout);
  boost::timer tmr;

  while (itext >> fsa) {

    //cerr << "sentence #" << sentenceno <<  "/" << total << endl;

    if (! fsa.empty()) {

      if (MODE == TRANSDUCTION) {

        totalmatches += rtn_transduct(fsa, parser, os, transmode, outputmode);

      } else { // concordance mode : only output sentences with matching sequences

        ostringstream oss;
        int nbmatches = rtn_transduct(fsa, parser, oss, transmode, outputmode);
        if (nbmatches) { os << oss.str(); }

        totalmatches += nbmatches;
      }
    }

    ++sentenceno;
    ++show_progress;
  }

  cout << "done. " << sentenceno << "/" << total << " sentences parsed.\n"
    << "found " << totalmatches << " matching sequences, " << tmr.elapsed() << "s.\n";

  if (outputmode == HTML) {
    os << "<br/>\n" << totalmatches << " matching sequences\n";
  }


  if (outputmode == HTML) { os << "</body>\n</html>\n"; }

  cout << "result stored in " << opath.string() << endl;

  xmlCleanupMemory();
  return 0;

} catch (exception & e) {

  cerr << "exception caught :" << e.what() << endl; exit(1);

} catch (...) { cerr << "caught an OVNI?\n"; exit(1); }

