#ifndef _LINGDEF_H_
#define _LINGDEF_H_

#include <iostream>
#include <string>
#include <vector>
#include <map>
#include <set>

#include <boost/filesystem/path.hpp>

#include <outilex/xml.h>

#include <outilex/attr_type.h>


/* attribute definition */

/*
 * attribute definition
 * contains its name and its type
 * e.g : name = transitive, type = bool
 *       name = gender,     type = enum (m, f)
 *       name = sem,        type = semtree
 *       etc.
 */

class feat_set;

class ling_def;
class pos_def;

class attr_def {

public:
  std::string  name;
  attr_type  * type;
  int def_value;
  bool shortcut;


public:

  attr_def(const std::string & _name, attr_type * _type, int def = 0)
    : name(_name), type(_type), def_value(def), shortcut(false) {}

  attr_def(xmlNodePtr node, ling_def * lingdef) { read_XML(node, lingdef); }


  const std::string & get_name() const { return name; }
  attr_type * get_type() const { return type; }

  int get_default_value() const { return def_value; }
  bool do_shortcut() const { return shortcut; }

  void register_shortcut_values(pos_def * pos);
  bool match(int v, const feat_set & fs) const;


  int get_value(const std::string & txt) const { return type->get_value(txt); }
  std::string get_value_name(int val) const { return type->get_val_text(val); }

  void dump_feat_val(int val, std::ostream & os) {
    type->dump_feat_val(get_name(), val, do_shortcut(), os);
  }

  void read_XML(xmlNodePtr node, ling_def * lingdef);
};


/* internal type */

extern attr_def * void_attr_def;


/* unknow attributes */
extern std::set<std::string> unknow_attributes;


/* POS definition */


class pos_def {

public:
  std::string name;
  ling_def * lingdef;

  std::vector<attr_def *> attrs;
  std::map<std::string, int> attr_by_name;

  typedef  std::map<std::string, std::pair<attr_def *, int> > shortcut_map;
  shortcut_map shortcut_values;

public:
  pos_def(const std::string & name_, ling_def * ldef)
    : name(name_), lingdef(ldef), attrs(), attr_by_name() {}

  pos_def(xmlNodePtr node, ling_def * lingdef);
  ~pos_def();

  const std::string & get_name() const { return name; }
  ling_def * get_lingdef() const { return lingdef; }


  template<class OutputIterator>
  static void minus(const pos_def * a, const pos_def * b, OutputIterator out);

  int nb_attrs() const { return attrs.size(); }

  attr_def * get_attr(int idx) const { return attrs[idx]; }
  attr_def * get_attr(const std::string & attr_name) const; 
  int get_attr_idx(const std::string & attr_name) const;
  int get_attr_idx(attr_def * attr) const { return get_attr_idx(attr->get_name()); }


  void register_shortcut(const std::string & shortcut, attr_def * attr, int v);

  bool is_shortcut_value(const std::string & text) const;
  bool find_shortcut_value(const std::string & text, attr_def * & attr, int & v) const;

  bool get_featval(const std::string & text, int & idx, int & val) const;
  bool get_feat_set(const std::string & text, attr_def * & attr, feat_set & fs) const;

  void read_XML(xmlNodePtr node);
};

extern pos_def * const pos_unspec; /* == -1 */


inline const pos_def * inter(const pos_def * a, const pos_def * b) {

  if (a == b) { return a; }
  if (a == pos_unspec) { return b; }
  if (b == pos_unspec) { return a; }

  /* incompatible pos */
  return 0;
}

template<class OutputIterator>
void minus(const pos_def * a, const pos_def * b, OutputIterator out) {
  pos_def::minus(a, b, out);
}

inline bool intersect(const pos_def * a, const pos_def * b) {
  return ((a != NULL) && (b != NULL)) && ((a == b) || (a == pos_unspec) || (b == pos_unspec));
}

inline bool in(const pos_def * a, const pos_def * b) {
  return (a == NULL) || (b == pos_unspec) || (a == b); 
}

/* syntagm definition */

typedef enum {
  DIC_FEAT_TYPE,     // feat from dictionnary
  STRING_FEAT_TYPE,  // feat value is a litteral string
  SYNT_FEAT_TYPE,    // feat value is a position in the sentence fsa
  SYNTS_FEAT_TYPE,   // a list of pos in sentence fsa
  INVALID_FEAT_TYPE  // invalid feat type
} syntagm_feat_type;


class syntagm_def {

  std::string name;
  std::map<std::string, attr_def *> attrs;
  std::map<std::string, syntagm_feat_type> feats_type;


public:

  explicit syntagm_def(const std::string & n) : name(n), attrs(), feats_type() {}

  const std::string & get_name() const { return name; }

  attr_def * get_dic_attr(const std::string & attrname) const;

  bool get_feat_set(const std::string & text, attr_def * & attr, feat_set & fs) const;

  /* return the type associated to attrname
   * in case of a non declared attribute, fallback to string type
   * in case of a dic feat attribute, fill the attr parameter to the corresponding attr definition
   */
  syntagm_feat_type get_feat_type(const std::string & attrname, attr_def * & attr) const;
};


/* ling_def : linguistic definitions */


class ling_def {

  friend class pos_def;

  std::string name;

  pos_def * pos_punc;
  pos_def * pos_number;
  pos_def * pos_unknow;
  pos_def * pos_epsilon;
  pos_def * pos_lex;

  typedef std::map<std::string, attr_type *> attr_map;
  typedef std::map<std::string, int> pos_map;
  typedef std::map<std::string, syntagm_def *> syntagm_map;

  std::vector<pos_def *> POSs;

  attr_map attr_types;
  pos_map  name2pos;
  syntagm_map syntagms;

  void read_xml(xmlNode * node);

public:

  typedef std::vector<pos_def *>::const_iterator const_pos_iterator;

  ling_def(xmlNodePtr node);
  ling_def(const boost::filesystem::path & path);

  ~ling_def();

  attr_type * get_attr_type(const std::string & type_name);
  pos_def   * get_pos(const std::string & pos_name);
 
  void add_attr_type(attr_type * type);

  void register_pos(pos_def * pos) { register_pos(pos, pos->get_name()); }
  void register_pos(pos_def * pos, const std::string & name); // to register alias to a given pos

  syntagm_def * get_syntagm_def(const std::string & syntagm_name);
  // create a new (empty) syntagm def
  syntagm_def * add_syntagm_def(const std::string & syntagm_name);

  // specials pos
  pos_def * punc_pos()    { return pos_punc; }
  pos_def * number_pos()  { return pos_number; }
  pos_def * unknow_pos()  { return pos_unknow; }
  pos_def * epsilon_pos() { return pos_epsilon; }
  pos_def * lex_pos()     { return pos_lex;}

  const_pos_iterator pos_begin() { return POSs.begin() + 1; } // skip epsilon
  //const_pos_iterator word_pos_begin() { return POSs.begin() + 3; } // skip pnc & nb poss
  const_pos_iterator pos_end() { return POSs.end(); }
};


template<class OutputIterator>
void pos_def::minus(const pos_def * a, const pos_def * b, OutputIterator out) {

  if ((a == b) || (b == pos_unspec)) { return; }

  if ((a != pos_unspec) || (b == NULL)) { *out = a; ++out; return; }

  /* a == pos_unspec && b != pos_unspec && b != NULL
   *
   * we return the list of the POSs minus b
   */

  const std::vector<pos_def *> & POSs = b->lingdef->POSs;

  for (std::vector<pos_def *>::const_iterator it = POSs.begin(); it != POSs.end(); ++it) {
    if ((*it != b) && (*it != b->lingdef->pos_epsilon)) {
      *out = *it; ++out;
    }
  }
}

#endif

