#ifndef _FEAT_SET_H_
#define _FEAT_SET_H_

#include <cassert>

#include <outilex/lingdef.h>
#include <outilex/xml.h>


/* feat_set
 * a feat_set represent a set of value for a given attribute type
 */

extern attr_type * void_attribute_type;


class feat_set {
  
public:

  attr_type * type;
  mutable attr_val_set * val;

public:

  /* should not be used, for internal use only */

  feat_set(attr_type * type, attr_val_set * val);

  /* default constructor attribute def should always be specified
   * default to void attribute type
   */
  feat_set() : type(void_attribute_type), val(0) {}

  
  feat_set(attr_type * type);
  feat_set(attr_def * def);

  /* copy constructor */
  feat_set(const feat_set & fs);

  /* copy assignment */

  feat_set & operator=(const feat_set & fs);


  /* construct from an XML description */

  feat_set(xmlNodePtr node, ling_def * ldef);


  /* construct from a attr_def * and a int or string value */

  feat_set(attr_type * attr, int value);
  feat_set(attr_type * attr, const std::string & value); // 'unset|masucline'
  feat_set(attr_def * attr, const std::string & value);

  feat_set & operator=(const std::string & value);


  ~feat_set() {
    if (type != void_attribute_type && type != 0) { delete_val(); }
  }


  /* empty the set */
  void clear();

  /* set the set to 'unset' special value */
  void unset();

  /* set the set to 'set' (everythin but 'unset') */
  void set();


  void set(attr_type * type, int v);
  void set(attr_type * type, const std::string & v);

  /* return true if empty set (match with no value)
   */
  bool empty() const;

  /* return true if the set is unspec (i.e. can match with any value)
   */
  bool is_unspec() const;


  /* return true if the set is not empty, false otherwise
   */
  operator const void*() const { return empty() ? 0 : this; }

  bool operator!() const { return empty(); }


  /* ensemblist operation */

  /* subset */
  static bool in(const feat_set & a, const feat_set & b);

  bool intersect(const feat_set & b) const;


  /* intersection */
  void set_inter(const feat_set & a, const feat_set & b);
  void set_inter(const feat_set & b);

  /* union */
  void set_union(const feat_set & a, const feat_set & b);
  void set_union(const feat_set & b);

  /* set difference */
  void set_minus(const feat_set & a, const feat_set & b);
  void set_minus(const feat_set & b);

  void operator&=(const feat_set & b) { set_inter(b); }
  void operator|=(const feat_set & b) { set_union(b); }
  void operator-=(const feat_set & b) { set_minus(b); }


  /* comparaison operation */

  bool operator==(const feat_set & b) const;
  bool operator!=(const feat_set & b) const {
    return ! (*this == b);
  }


  /* matche with feat value */
  bool match(int v) const;

  /* serialisation */

  /* <featset type='gender_type' value='unset|masculine'/> */

  void read_XML(xmlNodePtr node, ling_def * ldef);
  void write_XML(xmlwriter & os) const;
 
  /* 'unset|masculine' */

  void read_val(const std::string & str, attr_type * type);
  void dump_val(std::ostream & os) const;

  std::string get_val_text() const;


  /* gender_type:unset|masculine */

  static inline const char * xml_name() { return "featset"; }
  void read_text(const std::string & str, ling_def * ldef);
  void dump_text(std::ostream & os) const;

protected:
  // called from destructor
  void delete_val();
};



inline bool in(const feat_set & a, const feat_set & b) {
  return feat_set::in(a, b);
}

inline bool inter(const feat_set & a, const feat_set & b, feat_set & res) {
  res.set_inter(a, b);
  return ! res.empty();
}

inline feat_set inter(const feat_set & a, const feat_set & b) {
  feat_set res;
  res.set_inter(a, b);
  return res;
}

inline bool union_(const feat_set & a, const feat_set & b, feat_set & res) {
  res.set_union(a, b);
  return ! res.empty();
}

inline feat_set union_(const feat_set & a, const feat_set & b) {
  feat_set res;
  res.set_union(a, b);
  return res;
}

inline bool minus(const feat_set & a, const feat_set & b, feat_set & res) {
  res.set_minus(a, b);
  return ! res.empty();
}


inline feat_set operator&(const feat_set & a, const feat_set & b) { return inter(a,b); }

inline feat_set operator|(const feat_set & a, const feat_set & b) { return union_(a, b); }

inline std::ostream & operator<<(std::ostream & os, const feat_set & fs) {
  fs.dump_text(os); return os;
}

#endif

