#include <outilex/lexical_mask.h>
#include <outilex/dic_lex_entry.h>
#include <outilex/lexical_entry.h>


using namespace std;


/* construct from a dic_lex_entry
 */

lexical_mask::lexical_mask(const dic_lex_entry & e, ling_def * ldef) {

  unspec();

  if (! e.form.empty())  { form.read_text(e.form);   }
  if (! e.lemma.empty()) { lemma.read_text(e.lemma); }

  if (e.POS.empty()) { return; }

    
  pos = ldef->get_pos(e.POS);
  if (pos == NULL) {
    if (unknow_attributes.find(e.POS) == unknow_attributes.end()) {
      unknow_attributes.insert(e.POS);
      cerr << "lexmask:: unknow POS : " << e.POS << '\n';
    }
    clear();
    return;
  }

  /* first look for defaults values */

  int nattrs = pos->nb_attrs();
  for (int i = 0; i < nattrs; ++i) {
    attr_def * attr = pos->get_attr(i);
    if (attr->get_default_value() != -1) { // if default value is specified set it
      feat_set fs(attr->get_type(), attr->get_default_value());
      feats[attr] = fs;
    }
  }

  for (map<string, string>::const_iterator it = e.feats.begin(), end = e.feats.end();
       it != end; ++it) {

    attr_def * attr = pos->get_attr(it->first);
    if (attr == NULL) {
      if (unknow_attributes.find(it->first) == unknow_attributes.end()) {
        unknow_attributes.insert(it->first);
        cerr << "lexmak:: unknow attribute: " + it->first << " (in POS " << pos->get_name() << ")\n";
      }
      continue;
    }
  
    feat_set fs(attr->get_type(), it->second);
  
    if (! fs.is_unspec()) {
      feats[attr] = fs;
    }
  }
}


bool operator==(const lexical_mask & a, const lexical_mask & b) {

  if (& a == & b) { return true; }

  if (! a && ! b) { return true; }

  if (a.pos != b.pos) { return false; }

  if (a.form != b.form) { return false; }

  if (a.lemma != b.lemma) { return false; }

  if (a.feats.size() != b.feats.size()) { return false; }

  return a.feats == b.feats;
}


bool lexical_mask::set_inter(const lexical_mask & a, const lexical_mask & b) {

  if (& a == this) { lexical_mask aa = a; return set_inter(aa, b); }
  if (& b == this) { lexical_mask bb = b; return set_inter(a, bb); }

  if (! a || ! b) { // if one mask is empty return the empty set
    clear(); return false;
  }

  pos = inter(a.pos, b.pos);

  if (pos == 0) { // incompatible POSs
    clear(); return false;
  }

  if (! inter(a.form, b.form, form)) { /*cerr << "! inter form\n"; */ clear(); return false; }

  if (! inter(a.lemma, b.lemma, lemma)) { /* CERR << "! inter lemma\n"; */ clear(); return false; }


  if (pos != pos_unspec) {

    if (a.pos == pos_unspec) {

      feats = b.feats;

    } else if (b.pos == pos_unspec) {

      feats = a.feats;

    } else { // a.pos == b.pos (!= pos_unspec) : compute feats intersection

      feats = a.feats;
      feats_map::iterator ita, itb;

      for (itb = b.feats.begin(); itb != b.feats.end(); ++itb) {

        attr_def * attr = (*itb).first;
 
        ita = feats.find(attr);

        if (ita == feats.end()) { // attr not specified in a

          feats.insert(*itb);

        } else { // attr specified in a and b, compute intersection

          feat_set & fs = ita->second;
          fs &= itb->second; // intersect feat_set

          if (! fs) { // empty intersection -> masks are not compatible
            clear(); return false;
          }
        }
      }
    }

  } else { feats.clear(); }

  return true; 
}


void lexical_mask::set_inter(const lexical_mask & b) {

  if (empty() || ! b) { // if one mask is empty return the empty set
    clear(); return;
  }

  const pos_def * npos = inter(pos, b.pos);

  if (npos == 0) { // incompatible POSs
    clear(); return;
  }

  form &= b.form;
  if (! form) { /*cerr << "! inter form\n"; */ clear(); return; }

  lemma &= b.lemma;
  if (! lemma) { /* CERR << "! inter lemma\n"; */ clear(); return; }


  if (npos != pos_unspec) {

    if (pos == pos_unspec) { // pos was previously unset, copy b.feat

      feats = b.feats;

    } else if (b.pos != pos_unspec) { // a.pos == b.pos (!= pos_unspec) : compute feats intersection

      feats_map::iterator ita, itb;

      for (itb = b.feats.begin(); itb != b.feats.end(); ++itb) {

        attr_def * attr = itb->first;
 
        ita = feats.find(attr);

        if (ita == feats.end()) { // attr not specified in a

          feats.insert(*itb);

        } else { // attr specified in a and b, compute intersection

          feat_set & fs = ita->second;
          fs &= itb->second; // intersect feat_set

          if (! fs) { // empty intersection -> masks are not compatible
            clear(); return;
          }
        }
      }
    }

  } else { feats.clear(); }

  pos = npos;
}


bool lexical_mask::intersect(const lexical_mask & m) const {
 
  if (empty() || m.empty()) { return false; }

  if (! ::intersect(pos, m.pos)) {
    return false;
  }

  if (! (form.intersect(m.form) && lemma.intersect(m.lemma))) {
    return false;
  }

  if (pos == m.pos && pos != pos_unspec) {

    /* check that for each attribute specified in a and b
     * the intersection of the corresponding feat_set is non empty
     */
 
    feats_map::const_iterator ita = feats.begin(), itb = m.feats.begin(),
      enda = feats.end(), endb = m.feats.end();
 
    while (ita != enda && itb != endb) {
      if (ita->first < itb->first) { ++ita; continue; }
      if (itb->first < ita->first) { ++itb; continue; }
      // attribute types matches
      if (! ita->second.intersect(itb->second)) { return false; }
      ++ita, ++itb;
    }
  }

  return true;
}

bool lexical_mask::in(const lexical_mask & m) const {
 
  if (empty()) { return true; }

  if (! ::in(pos, m.pos)) {
    return false;
  }

  if (! (::in(form, m.form) && ::in(lemma, m.lemma))) {
    return false;
  }

  if (m.pos != pos_unspec) { // assert(pos == m.pos)
    
    /* check that for each attr where a feat_set b is specified in m,
     * a feat_set a is also specified in *this and (a in b)
     */

    for (feats_map::const_iterator itb = m.feats.begin(); itb != m.feats.end(); ++itb) {
      
      feats_map::const_iterator ita = feats.find((*itb).first);
      
      if ((ita == feats.end()) || (! ::in((*ita).second, (*itb).second))) { return false; }
    }
  }

  return true;
}


bool lexical_mask::match(const lexical_entry & e) const {

  if (pos != pos_unspec && pos != e.pos) { return false; }

  if (! (::match(e.form, form) && ::match(e.lemma, lemma))) { // forms don't match
    return false;
  }


  /* check that features are compatible */

  if (pos != pos_unspec) {
    for (feats_map::const_iterator it = feats.begin(); it != feats.end(); ++it) {

      attr_def * attr = it->first;
      const feat_set & fs = it->second;
      int idx = pos->get_attr_idx(attr);

      if (! fs.match(e.feats[idx])) { return false; } 
    }
  }

  /* match */
  return true;
}

