#include <string>

#include <outilex/xml.h>
#include <outilex/stringtok.h>

#include <outilex/enum_attr.h>

using namespace std;




/* Implementation of enumeration attribute type
 */


void enum_attr_type::read_XML(xmlNodePtr node) {

  value_name.clear(); name_to_value.clear();

  char * text = xmlGetProp(node, "name");
  name =  text;
  xmlFree(text);

  /* first add 'unset' value (always == 0) */

  int idx = 0;
  string v = "unset";

  value_name.push_back(v);
  name_to_value[v] = idx;
  idx++;

  node = node->xmlChildrenNode;

  while (node) {
    if (xmlStrcmp(node->name, (const xmlChar *) "value") == 0) {

      text = xmlGetProp(node, "name");
      v = text;

      value_name.push_back(v);
      name_to_value[v] = idx;

      xmlFree(text);

      if ((text = xmlGetProp(node, "alias"))) {
        vector<string> aliases;
        stringtok(text, ",", back_inserter(aliases));
        for (vector<string>::const_iterator it = aliases.begin(); it != aliases.end(); ++it) {
          name_to_value[*it] = idx;
        }
        xmlFree(text);
      }
      idx++;
    }

    node = node->next;
  }
}


/* enum attribute loader routines */

attr_type * enum_attr_loader(xmlNodePtr node) {
  return new enum_attr_type(node);
}

//static attr_type_registerer enum_registration("enum", enum_attr_loader);


const string & enum_attr_type::get_name() const { return name; }

namespace {

const int UNSET  = 1;
const int UNSPEC = ~UNSET;

} //namespace ""


attr_val_set * enum_attr_type::new_val_set() {
  return (attr_val_set *) UNSPEC;
}


attr_val_set * enum_attr_type::new_val_set(int val) {

  int res;
  if (val == -1) { // everything but unset (= 0)
    res = UNSPEC;
  } else {
    res = 1 << val;
  }
  
  return reinterpret_cast<attr_val_set*>(res);
}

attr_val_set * enum_attr_type::new_val_set(const attr_val_set * vs) {
  return const_cast<attr_val_set*>(vs);
}

attr_val_set * enum_attr_type::new_val_set(const string & str) {

  int res = 0;

  bool neg = false;

  if (str.empty()) { return (attr_val_set *) 0; }

  if (str[0] == '!') { 
    neg = true;
    // set all but 'unset' to true
    res = UNSPEC;
  }

  vector<string> vec;
  stringtok(str, "|!", back_inserter(vec));

  for (int i = 0; i < vec.size(); ++i) {

    map<string, int>::iterator it = name_to_value.find(vec[i]);

    if (it == name_to_value.end()) {
      throw xml_parse_error("in enum attribute: " + name + ": unknow value '" + vec[i] + "'");
    }

    if (neg) {
      res &= ~(1 << it->second);
    } else {
      res |= (1 << it->second);
    }
  }

  return (attr_val_set *)(res);
}


void enum_attr_type::del_val_set(attr_val_set * vs) { }

void enum_attr_type::copy_val_set(attr_val_set *& dest, const attr_val_set * src) {
  dest = const_cast<attr_val_set*>(src);
}


bool enum_attr_type::is_val_set_empty(const attr_val_set * vs) {
  return reinterpret_cast<int>(vs) == 0;
}

bool enum_attr_type::is_val_set_unspec(const attr_val_set * vs) {
  return (reinterpret_cast<int>(vs) & UNSPEC) == UNSPEC;
}


void enum_attr_type::clear_val_set(attr_val_set *& vs) {
  reinterpret_cast<int&>(vs) = 0;
}

void enum_attr_type::unset_val_set(attr_val_set *& vs) {
  reinterpret_cast<int&>(vs) = UNSET;
}

void enum_attr_type::set_val_set(attr_val_set *& vs) {
  reinterpret_cast<int&>(vs) = UNSPEC;
}



bool enum_attr_type::in(const attr_val_set * a, const attr_val_set * b) {
  return (reinterpret_cast<int>(a) & reinterpret_cast<int>(b)) == reinterpret_cast<int>(a);
}

bool enum_attr_type::intersect(const attr_val_set * a, const attr_val_set * b) {
  return (reinterpret_cast<int>(a) & reinterpret_cast<int>(b));
}


attr_val_set * enum_attr_type::inter(const attr_val_set * a, const attr_val_set * b) {
  return reinterpret_cast<attr_val_set*>(reinterpret_cast<int>(a) & reinterpret_cast<int>(b));
  //return a & b;
}

attr_val_set * enum_attr_type::union_(const attr_val_set * a, const attr_val_set * b) {
  return reinterpret_cast<attr_val_set*>(reinterpret_cast<int>(a) | reinterpret_cast<int>(b));
  //return a | b;
}

attr_val_set * enum_attr_type::minus(const attr_val_set * a, const attr_val_set * b) {
  return reinterpret_cast<attr_val_set*>(reinterpret_cast<int>(a) & ~(reinterpret_cast<int>(b)));
  //return a & ~b;
}


bool enum_attr_type::equal(const attr_val_set * a, const attr_val_set * b) {
  return a == b;
}

bool enum_attr_type::match(int v, const attr_val_set * vs) {
  return reinterpret_cast<int>(vs) & (1 << v);
}


void enum_attr_type::register_shortcut_values(attr_def * attr, pos_def * pos) {

  string shortcut;

  for (map<string, int>::const_iterator it = name_to_value.begin();
       it != name_to_value.end(); ++it) {

    if ((*it).first != "unset") {
      shortcut = '+' + (*it).first;
      pos->register_shortcut(shortcut, attr, it->second);
    }
  }
}



void enum_attr_type::dump_val_set_text(const attr_val_set * vs, std::ostream & os) const {

  int evs = reinterpret_cast<int>(vs);

  bool already = false;

  for (int i = 0; i < nb_values(); ++i) {
    if (evs & (1 << i)) {
      if (already) { os << '|'; }
      os << value_name[i];
      already = true;
    }
  }
}

string enum_attr_type::get_val_set_text(const attr_val_set * vs) const {

  string res;
  int evs = reinterpret_cast<int>(vs);

  bool already = false;

  for (int i = 0; i < nb_values(); ++i) {
    if (evs & (1 << i)) {
      if (already) { res += '|'; }
      res += value_name[i];
      already = true;
    }
  }
  return res;
}


string enum_attr_type::get_val_text(int val) const { return value_name[val]; }

void enum_attr_type::dump_val_text(int val, ostream & os) const { os << value_name[val]; }


void enum_attr_type::dump_feat_val(const std::string & attrname,
                                   int val, bool shortcut, std::ostream & os) const {
  os << '+';
  if (! shortcut) {
    os <<  attrname << '=';
  }
  os << value_name[val];
}


void enum_attr_type::dump_feat_val_set(const std::string & attrname, const attr_val_set * vs,
                                       bool shortcut, std::ostream & os) const {

  if (! shortcut) {
    os << '+' << attrname << '='; dump_val_set_text(vs, os);
    return;
  }

  int evs = reinterpret_cast<int>(vs);

  for (int i = 0; i < nb_values(); ++i) {
    if (evs & (1 << i)) {
      os << '+' << value_name[i];
    }
  }
}


int enum_attr_type::get_value(const std::string & name) const {
  map<string, int>::const_iterator it = name_to_value.find(name);
  if (it == name_to_value.end()) { return -1; }
  return it->second;
}

