#include "syntactic_analysis.h"
#include "utilities.h"
#include <stdlib.h>

int sa_n_free = 0;
int sa_n_malloc = 0;

void* sa_malloc(int n){
  sa_n_malloc++;
  return (void *)malloc(n);
}

void sa_pfree(void *ptr){
  if(ptr == NULL) return;
  sa_n_free++;
  free(ptr);
  ptr = NULL;
}

int sa_print_n_malloc(){
  printf("sa_malloc=%d\n",sa_n_malloc);
  return sa_n_malloc;
}

int sa_print_n_free(){
  printf("sa_free=%d\n",sa_n_free);
  return sa_n_free;
}

struct syntactic_tree* sa_create_tree(){
  int i;
  struct syntactic_tree *tree = (Syntactic_tree)sa_malloc(sizeof(struct syntactic_tree));
  tree->size = 0;
  tree->MAX = SA_TREE_SIZE;
  Syntactic_node* tab = (Syntactic_node*)sa_malloc(tree->MAX*sizeof(Syntactic_node));
  for(i = 0 ; i < tree->MAX ; i++){
    tab[i] = NULL;
  }  
  tree->tab = tab;
  return tree;
}


int sa_add_tree_node(Syntactic_tree *tree,struct syntactic_node *snode){
  int i,size;
  size = (*tree)->size;
  
  if(size == (*tree)->MAX){
    (*tree)->MAX = size + SA_TREE_SIZE;
    (*tree)->tab = realloc((*tree)->tab,(*tree)->MAX*sizeof(Syntactic_node));
    if((*tree)->tab == NULL) util_error("sa_add_tree_node","cannot add a new node");
    for(i = size ; i < (*tree)->MAX ; i++){
      (*tree)->tab[i] = NULL;
    }
  }
  (*tree)->tab[(*tree)->size] = snode;
  ((*tree)->size)++;
  return (*tree)->size - 1;
}


struct syntactic_node *sa_create_node(){
  struct syntactic_node *node = (struct syntactic_node *)sa_malloc(sizeof(struct syntactic_node));
  if(node == NULL) util_error("sa_create_node","Cannot create node");
  
  node->pattern = -1;
  node->sn_list = NULL;
  node->features = hash_init(SA_HASH_SIZE);
  
  return node;
}


void sa_init_tree(Syntactic_tree *tree){
  *tree = sa_create_tree();
  sa_add_tree_node(tree,sa_create_node());
}

void sa_init_tree2(Syntactic_tree *tree){
  *tree = sa_create_tree();
}

void sa_add_tree_transition(Syntactic_tree *tr,int src,int dest){
  Syntactic_node node = sa_get_tree_node(*tr,src);
  if(node == NULL) util_error("sa_add_tree_transition","source node does not exit");
  sa_add_node_sn_list_element(&node,dest);
}

void sa_add_node_feature(Syntactic_node *node,int type,int value){
  if(*node == NULL) util_error("sa_add_node_feature","node should not be null");
  hash_add_element(&((*node)->features),type,value,-1);
}

void sa_set_node_pattern(Syntactic_node *node,int p){
  if(*node == NULL) util_error("sa_add_node_pattern","node should not be null");
  (*node)->pattern = p;  
}

void sa_add_node_sn_list_element(Syntactic_node *node,int sn_node){
  if(*node == NULL) util_error("sa_add_node_sn_list_element","node should not be null");
  ll_i_insert_at_tail(&((*node)->sn_list),sn_node);    
}

void sa_free_node(Syntactic_node *node){
  if(*node == NULL) return;
  ll_i_free(&((*node)->sn_list));
  hash_free((*node)->features);
  sa_pfree(*node);
}

void sa_free_tree(Syntactic_tree *tree){
  int i;
  if(*tree == NULL) return;
  for(i = 0 ; i < (*tree)->size ; i++){
    sa_free_node(&((*tree)->tab[i]));
  }
  if((*tree)->tab != NULL) sa_pfree((*tree)->tab);
  sa_pfree(*tree);
}


int sa_print_features(Hashtable ht,Lexicon lex,int header){
  Llist_i l = hash_get_keys(ht);
  int key,value;
  struct ll_i_cell *ptr;

  if(l == NULL) return 0;
  ptr = l->head;
  printf("<");
  ustring_print(lex_get_element(lex,hash_get_value(ht,header)));
  while(ptr != NULL){
    key = ptr->value;
    if(key != header){
      printf(" ");
      ustring_print(lex_get_element(lex,key));
      printf("=");
      value = hash_get_value(ht,key);
      if(value >= 0){
	ustring_print(lex_get_element(lex,value));
      }
      else{
	printf("%d",-value-1);
      }
    }
    ptr = ptr->next;
  }
  printf(">");
  ll_i_free(&l);
  return 1;
}

void sa_print_pattern(int pattern,Lexicon lex){
  //  Ustring u;
  if(pattern == -1) return;
  // u = lex_get_element(lex,pattern);
  //if(u == NULL) return;
  //  ustring_print(u);
  printf("%d",pattern);
}

void sa_print_end_feature(Ustring u){
  printf("</");
  ustring_print(u);
  printf(">");
}


void sa_print_tree_rec(Syntactic_tree tree,int index, Lexicon lex,int header){
  Syntactic_node node = sa_get_tree_node(tree,index);
  Llist_i l = NULL;
  int found;
  struct ll_i_cell *ptr;

  if(node == NULL) return;
  found = sa_print_features(node->features,lex,header);
  l = node->sn_list;
  if(l != NULL){
    ptr = l->head;
    while(ptr != NULL){
      sa_print_tree_rec(tree,ptr->value,lex,header);
      ptr = ptr->next;
    }
  }
  sa_print_pattern(node->pattern,lex);  
  if(found){
    sa_print_end_feature(lex_get_element(lex,hash_get_value(node->features,header)));
    // printf("\n");
  }
}


void sa_print_tree(Syntactic_tree tree, Lexicon lex,int header){  
  if(tree == NULL) return;
  sa_print_tree_rec(tree,0,lex,header);
  printf("\n");
}


Syntactic_node sa_get_tree_node(Syntactic_tree tree,int index){
  if(tree == NULL) return NULL;
  if(index < 0) return NULL;
  if(tree->tab == NULL) return NULL;
  if(tree->MAX <= index) return NULL;
  return tree->tab[index];
}


int sa_get_feature_value(Syntactic_tree tree,int feature_key,int node_index){
  Syntactic_node node = sa_get_tree_node(tree,node_index);
  if(node == NULL) return 0;
  return hash_get_value(node->features,feature_key);
}
