#include <stdio.h>
#include <malloc.h>
#include "ling_representation.h"
#include "ustring.h"
#include "utilities.h"
#include "tab_u.h"
#include "tab2_u.h"

int ling_n_free = 0;
int ling_n_malloc = 0;

void* ling_malloc(int n){
  ling_n_malloc++;
  return (void *)malloc(n);
}

void ling_pfree(void *ptr){
  if(ptr == NULL) return;
  ling_n_free++;
  free(ptr);
  ptr = NULL;
}

int ling_print_n_malloc(){
  printf("ling_malloc=%d\n",ling_n_malloc);
  return ling_n_malloc;
}

int ling_print_n_free(){
  printf("ling_free=%d\n",ling_n_free);
  return ling_n_free;
}

Ling_rep ling_init(){
  Ling_rep lr = (Ling_rep)ling_malloc(sizeof(struct ling_rep));
  if(lr == NULL) util_error("ling_init","cannot initialize ling_rep");
  lr->lexicon = NULL;
  lr->b_features = NULL;
  lr->ling_units = NULL;
  return lr;
}


int add_boolean_feature(Ling_rep *lr,Ustring type,Ustring label){
  int id_type,id_label;

  if(*lr == NULL){
    (*lr) = ling_init();
  }
  // add in lexicon
  id_type = lex2_add_element(&((*lr)->lexicon),type);
  id_label = lex2_add_element(&((*lr)->lexicon),label);
  //add in b_features
  return tab_feat_add_element(&((*lr)->b_features),id_type,id_label);
}



Ustring ling_get_word_from_dico_line(Ustring u,int *i){
  Ustring word = NULL;
  while(u[*i] != 0 && u[*i] != ','){
    ustring_append(&word,u[*i]);
    (*i)++;
  }
  if(u[*i] == 0){
    ustring_free(word);
    return NULL;
  }
  (*i)++;
  return word;
}

Ustring ling_get_root_from_dico_line(Ustring u,Ustring word,int *i){
  Ustring root = NULL;
  if(word == NULL) return NULL;
  if(u[*i] == '.'){
    root = ustring_allocation(word);
  }
  else{ 
    while((u[*i] != 0) && (u[*i] != '.')){
      ustring_append(&root,u[*i]);
      (*i)++;
    }
    if(u[*i] == 0){
      ustring_free(root);
      return NULL;
    }
  }

  (*i)++;
  return root;
}

Ustring ling_get_pos_from_dico_line(Ustring u,int *i){
  Ustring pos = NULL;
  if(u[*i] == 0) return NULL;
  while((u[*i] != 0)&&(u[*i] != ':')&& (u[*i] != '+')){
    ustring_append(&pos,u[*i]);
    (*i)++;
  }
  return pos;
}

Tab_u ling_get_sem_features_from_dico_line(Ustring u,int *i){
  Tab_u sem_feat = NULL;
  int cnt = 0;
  Ustring temp;

  while(u[*i] == '+'){
    (*i)++;
    temp = NULL;
    while((u[*i] != 0) && (u[*i] != ':')&& (u[*i] != '+')){
      ustring_append(&temp,u[*i]);
      (*i)++; 
    }
    tab_u_assign_value(&sem_feat,cnt,temp);
    ustring_free(temp);
    cnt++;
  }
  return sem_feat;
}

Tab2_u ling_get_flex_features_from_dico_line(Ustring u,int *i){
  Tab2_u flex_feat = NULL;
  int n_r = 0,n_c = 0;
  Ustring temp;

  while(u[*i] == ':'){
    (*i)++;
    n_c = 0;
    while((u[*i] != 0) && (u[*i] != ':')){
      temp = NULL;
      ustring_append(&temp,u[*i]);
      tab2_u_assign_value(&flex_feat,n_r,n_c,temp);
      ustring_free(temp);
      (*i)++;
      n_c++;
    }
    n_r++;
  }
  return flex_feat;
}

void ling_insert_boolean_feature(Ling_rep *lr, Ustring u1, 
				 Ustring u2, Llist_i *l){
  int b_id;
  b_id = add_boolean_feature(lr,u1,u2);
  ll_i_insert_sorted(l,b_id + 1);
}

void ling_flex_split_and_ling_units(Ling_rep *lr,Ustring ff,Llist_i l,Tab2_u flex_feat){
  int i,j,n_r,n_c;
  Llist_i temp;
  Ustring u;
  n_r = tab2_u_get_n_rows(flex_feat);
  for(i = 0 ; i < n_r ; i++){
    temp = ll_i_copy(l);
    n_c = tab2_u_get_n_columns(flex_feat,i);
    for(j = 0 ; j < n_c ; j++){      
      ling_insert_boolean_feature(lr,ff,tab2_u_get_value(flex_feat,i,j),&temp);
    }
    u = ustring_from_list(temp);
    //  lex2_add_element(&((*lr)->ling_units),u);
    ustring_free(u);
    ll_i_free(&temp);
  }
}

void  ling_create_ling_units(Ling_rep *lr,Ustring word,Ustring root,
			      Ustring pos,Tab_u sem_feat,
			      Tab2_u flex_feat,Tab_u types){
  int i,n;
  Llist_i l = NULL;
  
  ling_insert_boolean_feature(lr,tab_u_get_value(types,0),word,&l);
  ling_insert_boolean_feature(lr,tab_u_get_value(types,1),root,&l);
  ling_insert_boolean_feature(lr,tab_u_get_value(types,2),pos,&l);
  n = tab_u_get_n_elements(sem_feat);
  for(i = 0 ; i < n ; i++){
      ling_insert_boolean_feature(lr,tab_u_get_value(types,3),tab_u_get_value(sem_feat,i),&l);
  }
  ling_flex_split_and_ling_units(lr,tab_u_get_value(types,4),l,flex_feat);
  ll_i_free(&l);
}

Tab_u ling_init_types_tab(){
  Ustring w,r,p,sf,ff;
  Tab_u t = NULL;

  w =  ustring_from_string("_WORD");
  r = ustring_from_string("_ROOT");
  p = ustring_from_string("_POS");
  sf = ustring_from_string("_SEM");
  ff = ustring_from_string("_FLEX");
  
  tab_u_assign_value(&t,0,w);
  tab_u_assign_value(&t,1,r);
  tab_u_assign_value(&t,2,p);
  tab_u_assign_value(&t,3,sf);
  tab_u_assign_value(&t,4,ff);

  ustring_free(w);
  ustring_free(r);
  ustring_free(p);
  ustring_free(sf);
  ustring_free(ff);

  return t;
}



struct ling_rep *load_dico(char name[],int option){
  FILE *f;
  Ustring u,word= NULL,root = NULL,pos = NULL;
  Tab_u sem_feat = NULL,types;
  Tab2_u flex_feat = NULL;
  struct ling_rep *lr = NULL;
  int i;

  f = fopen(name,"r");
  if(f == NULL){
    util_error("load_dico","cannot read file");
  }
  if(!option) ustring_read_uchar(f,option);//read unicode special character
  
  types = ling_init_types_tab();
  while((u = ustring_get_line(f,option)) != NULL){
    i = 0;
    word = ling_get_word_from_dico_line(u,&i);
    root = ling_get_root_from_dico_line(u,word,&i);
    pos = ling_get_pos_from_dico_line(u,&i); 
    sem_feat = ling_get_sem_features_from_dico_line(u,&i);
    flex_feat = ling_get_flex_features_from_dico_line(u,&i);
    ling_create_ling_units(&lr,word,root,pos,sem_feat,flex_feat,types);
    
    ustring_free(u);
    ustring_free(root);
    ustring_free(word);
    ustring_free(pos);
    tab_u_free(&sem_feat);
    tab2_u_free(&flex_feat);
  }
  tab_u_free(&types);
  fclose(f);
  return lr;
}

void ling_print(struct ling_rep *lr){
  if(lr == NULL){
    printf("[empty linguistic representation]\n");
    return;
  }
  printf("LEXICON\n======\n");
  lex2_print(lr->lexicon);  
  printf("\n======\n");  
  printf("BOOLEAN FEATURES\n****\n");
  tab_feat_print(lr->b_features);  
  printf("\n****\n");
  printf("LING UNITS\n-----\n");
  lex2_println2(lr->ling_units);  
  printf("\n------\n");
}

void ling_free(Ling_rep *lr){
  if(*lr == NULL) return;
  lex2_free(&((*lr)->lexicon));
  tab_feat_free(&((*lr)->b_features));
  lex2_free(&((*lr)->ling_units));
  ling_pfree(*lr);
}
