#include <malloc.h>
#include <math.h>
#include "ustring.h"
#include "tree.h"
#include "fst2.h"
#include "utilities.h"


int fst2_n_free = 0;
int fst2_n_malloc = 0;

void* fst2_malloc(int n){
  fst2_n_malloc++;
  return (void *)malloc(n);
}

void fst2_pfree(void *ptr){
  fst2_n_free++;
  free(ptr);
}

int fst2_print_n_malloc(){
  printf("fst2_malloc=%d\n",fst2_n_malloc);
  return fst2_n_malloc;
}

int fst2_print_n_free(){
  printf("fst2_free=%d\n",fst2_n_free);
  return fst2_n_free;
}

void fst2_initialize_context(struct fst2_context *context,int option){

  context->n_automata = 0;
  context->n_terminals = 0;
  context->n_tags = 0;
  context->tree_names = tree_init();
  context->tree_terminals = tree_init();
  context->option = option;
  context->graph_names = NULL;
  context->terminals = NULL;
  context->tags = NULL;
  context->tag_analyses = NULL;
  context->inputs = NULL;
  context->outputs = NULL;
  context->weights = NULL;
}

int fst2_get_n_automata(FILE *f,int option){
  Ustring u = ustring_get_line(f,option);
  int i = 0,n_auto;
  if(!option) i = 1;
  if(u == NULL) return -1;
  n_auto = ustring_get_next_int(u,&i);
  ustring_free(u);
  return n_auto;
}

Automaton fst2_load_one_graph(FILE *f,struct fst2_context *context,int option,int *index_automaton,Ustring *graph_name){
  Ustring u;
  State st;
  Automaton aut = NULL;
  Llist ll_auxiliary = NULL;
  Llist ll_terminal = NULL;
  int k = 0,n_states = 0,tag,arr,pos;

  u = ustring_get_line(f,option);
  *index_automaton = ustring_get_next_int(u,&k);
  *index_automaton = -(*index_automaton)-1;
  *graph_name = ustring_get_substr(u,k+1,ustring_length(u));
  // ustring_println(*graph_name);
  ustring_free(u);
  while( ((u =  ustring_get_line(f,option))!= NULL) && (u[0] != 'f')){
    ll_auxiliary = NULL;
    ll_terminal = NULL;
    st = auto_init_state();
    pos = 1;
    if(n_states == 0) auto_set_initial_state(st); 
    if(u[0] == 't') auto_set_final_state(st);
    while((tag = ustring_get_next_int(u,&pos)) != 0){
      arr = ustring_get_next_int(u,&pos);
      if(tag < 0){
	tag = -tag -1;
	ll_insert_sorted(&ll_auxiliary,tag,arr,0);
      }
      else{
	ll_insert_sorted(&ll_terminal,tag,arr,0);
      }
    }
    auto_copy_transitions(&st, ll_terminal,ll_auxiliary);
    ustring_free(u);
    ll_free(&ll_auxiliary);
    ll_free(&ll_terminal);
    auto_add_state(&aut,n_states,st);
    n_states++;
  }
  ustring_free(u);

  return aut;
}


void fst2_free_context(struct fst2_context *context){

  tab_u_free(&(context->graph_names));
  tab_u_free(&(context->terminals));
  tab_u_free(&(context->tags));
  tab_l_free(&(context->tag_analyses));
  tab_i_free(&(context->inputs));
  tab_i_free(&(context->outputs));
  tab_d_free(&(context->weights));
  tree_free(&(context->tree_names));
  tree_free(&(context->tree_terminals));
}

Llist fst2_get_tag_analysis(Ustring u, Tokens *tok){
  return NULL;
}


void fst2_add_element(Ustring u, struct fst2_context *context, int* n,int is_input,int index,Tokens *tok){
  int num = *n;
  Llist tag_analysis;

  if(u == NULL) return;
  if((num = tree_word_exists(context->tree_terminals,u)) == -1){
    tree_insert_word(&(context->tree_terminals),u,*n);
    tab_u_assign_value(&(context->tags),*n,u);
    num = *n;
    (*n)++;
    context->n_tags = *n;
  }
  if(is_input == 1){
    tab_i_assign_value2(&(context->inputs),index,num,-1);
    if(tok != NULL) tag_analysis = fst2_get_tag_analysis(u,tok);
  }
  else{
    tab_i_assign_value2(&(context->outputs),index,num,-1);
  }

}


int fst2_get_terminals(FILE *f,struct fst2_context *context,int option,Tokens *tok){
  Ustring u,v,w;
  Tab_u temp = NULL;
  int i,n_tags;
  double weight;
  
  i = 0;
  n_tags = 0;
  context->n_terminals = 0;
  while((u = ustring_get_line(f,option)) && (u[0] != 'f')){
    v = ustring_get_substr(u,1,ustring_length(u));
    tab_u_assign_value(&(context->terminals),i,v);    
    temp = ustring_split('/',v);
    fst2_add_element(tab_u_get_value(temp,0),context,&n_tags,1,i,tok); //1 = for input_tags
    fst2_add_element(tab_u_get_value(temp,1),context,&n_tags,0,i,tok);//0 = for output_tags    
    w = tab_u_get_value(temp,2);
    if(ustring_length(w) == 0) weight = 1.0;
    else weight = ustring_to_double(w);
    tab_d_assign_value2(&(context->weights),i,weight,1.0);
    i++;
    (context->n_terminals)++;    
    ustring_free(u);
    ustring_free(v);
    tab_u_free(&temp);
  }
  ustring_free(u);
  return 0;
}

void fst2_modify_terminals_in_state(State *st,struct fst2_context *context){
  int i,n;
  if((*st)->terminal_trans == NULL) return;
  for(i = 0 ; i < (*st)->terminal_trans->n_transitions ; i++){
    n = tab_i_get_value((*st)->terminal_trans->input_tags,i);
    tab_i_assign_value(&((*st)->terminal_trans->input_tags),i,tab_i_get_value(context->inputs,n));
    tab_i_assign_value(&((*st)->terminal_trans->output_tags),i,tab_i_get_value(context->outputs,n));
    tab_d_assign_value2(&((*st)->terminal_trans->weights),i,tab_d_get_value(context->weights,n),1.0);
  }
}

void fst2_modify_terminals_in_automaton(Automaton *aut,struct fst2_context *context){
  int i;
  State st;

   for(i = 0 ; i < (*aut)->n_states ; i++){
     st = auto_get_state(*aut,i);
     fst2_modify_terminals_in_state(&st,context);
  }
}


void fst2_modify_terminals_in_grammar(Grammar *gram,struct fst2_context *context){
  int i;
  Automaton aut;

  //  if(*gram == NULL)util_error("fst2_modify_terminals_in_grammar","*grm should not be null");
  for(i = 0 ; i < (*gram)->n_automata ; i++){
    aut = gram_get_automaton(*gram,i);
    fst2_modify_terminals_in_automaton(&aut,context);
  }
  //update terminal alphabet
  gram_set_n_terminals(*gram,context->n_tags);
  for(i = 0 ; i < context->n_tags ; i++){
    tab_u_assign_value(&((*gram)->terminals),i,tab_u_get_value(context->tags,i));
  }
}


void fst2_init_aux_outputs_in_state(State *st){
  int i;
  if((*st)->auxiliary_trans == NULL) return;
  for(i = 0 ; i < (*st)->auxiliary_trans->n_transitions ; i++){
    tab_i_assign_value(&((*st)->auxiliary_trans->output_tags),i,-1);
    tab_d_assign_value(&((*st)->auxiliary_trans->weights),i,1.0);
  }
}

void fst2_init_aux_outputs_in_automaton(Automaton *aut){
  int i;
  State st;
  
   for(i = 0 ; i < (*aut)->n_states ; i++){
     st = auto_get_state(*aut,i);
     fst2_init_aux_outputs_in_state(&st);
  }
}


void fst2_init_aux_outputs_in_grammar(Grammar *gram){
  int i;
  Automaton aut;

  for(i = 0 ; i < (*gram)->n_automata ; i++){
    aut = gram_get_automaton(*gram,i);
    fst2_init_aux_outputs_in_automaton(&aut);
  } 
}


// load a fst2 graph 
//option = 0 => unitex format
//option = 1 => intex format

Grammar fst2_load_graph(char *name,int is_intex_format, Tokens *tok){
  struct fst2_context context;
  Automaton aut;
  Ustring graphname;
  Grammar gram = NULL;
  int i,k;
  FILE *f;
  
  f = fopen(name,"r");
  if(f == NULL){
    printf("Cannot open fst2-grammar %s\n",name);
    return NULL;
  }  
  fst2_initialize_context(&context,is_intex_format);
  context.n_automata = fst2_get_n_automata(f,is_intex_format);
  //  printf("%d\n",context.n_automata);
  i = 0;
  if(context.n_automata <= 0) util_error("fst2_load_graph","number loading error");
  while(i < context.n_automata){
    aut = fst2_load_one_graph(f,&context,is_intex_format,&k,&graphname);
    gram_add_automaton(&gram,k,aut);
    gram_add_graph_name(&gram,k,graphname);
    ustring_free(graphname);
    i++;
  }
  fst2_get_terminals(f,&context,is_intex_format,tok);
  fst2_modify_terminals_in_grammar(&gram,&context);
  fst2_init_aux_outputs_in_grammar(&gram); 
 /*  for(i = 0 ; i < context.n_terminals ; i++){
    printf("+++\n%d::tag=",i);
    ustring_println(context.terminals[i]);
    printf("in=");
    ustring_println(context.tags[context.inputs[i]]);
    printf("out=");
    if(context.outputs[i] != -1) ustring_println(context.tags[context.outputs[i]]);
    printf("\n++++\n");
    }*/
  fclose(f);
  fst2_free_context(&context);
  return gram;
}
