#include <stdio.h>
#include <malloc.h>
#include "automata.h"
#include "utilities.h"

int auto_n_free = 0;
int auto_n_malloc = 0;

void* auto_malloc(int n){
  auto_n_malloc++;
  return (void *)malloc(n);
}

void auto_pfree(void *ptr){
  if(ptr == NULL) return;
  auto_n_free++;
  free(ptr);
  ptr = NULL;
}

int auto_print_n_malloc(){
  printf("auto_malloc=%d\n",auto_n_malloc);
  return auto_n_malloc;
}

int auto_print_n_free(){
  printf("auto_free=%d\n",auto_n_free);
  return auto_n_free;
}

void auto_free_transitions(Transitions *t){
  if(*t == NULL) return;
  tab_i_free(&((*t)->input_tags));
  tab_i_free(&((*t)->output_tags));
  tab_d_free(&((*t)->weights));
  tab_i_free(&((*t)->arr));
  auto_pfree(*t);
}

Transitions auto_get_terminal_transitions(State st){
  if(st == NULL) return NULL;
  return st->terminal_trans; 
}

Transitions auto_get_auxiliary_transitions(State st){
  if(st == NULL) return NULL;
  return st->auxiliary_trans; 
}
int auto_get_n_transitions(Transitions tr){
  if(tr == NULL) return 0;
  return tr->n_transitions;
}

void auto_free_state(State *st){
  if(*st == NULL) return;
  auto_free_transitions(&((*st)->terminal_trans));
  auto_free_transitions(&((*st)->auxiliary_trans)); 
  auto_pfree(*st);
}

void auto_free(Automaton *aut){
  long i,n_states;
  if(*aut == NULL) return;
  n_states = (*aut)->n_states;
  for(i = 0 ; i < n_states ; i++){
    auto_free_state(&((*aut)->states[i]));
  }
  auto_pfree((*aut)->states);
  auto_pfree(*aut);
}


int auto_get_n_states(Automaton aut){
  if(aut == NULL) return 0;
  return (aut->n_states);
}


Automaton auto_init_automaton(){
  Automaton aut = NULL;
  
  aut = (Automaton)auto_malloc(sizeof(struct automaton));
  if(aut == NULL) util_error("auto_init_automaton","allocation of new automaton");
  aut->n_states = 0;
  aut->states = NULL;
  aut->MAX = 0;
  return aut;
}

State * auto_state_allocation(Automaton *aut,int min_size){
  int start,i;
  if(*aut == NULL) util_error("auto_state_allocation","*aut is NULL");
  if(min_size <= (*aut)->MAX) return (*aut)->states;
  start = (*aut)->MAX;
  (*aut)->MAX = ((min_size/AUTO_STEP_ALLOC)+1)*AUTO_STEP_ALLOC;

  if((*aut)->states == NULL) 
    (*aut)->states = (State *)auto_malloc((*aut)->MAX * sizeof(State));
  else 
    (*aut)->states = (State *)realloc(*aut,(*aut)->MAX * sizeof(State));
  
  if((*aut)->states == NULL) 
    util_error("auto_state_allocation","allocation of *aut");
  
  //initialization of each state to NULL
  for(i = start ; i < (*aut)->MAX ; i++){
    (*aut)->states[i] = NULL;
  }
  return (*aut)->states;
}

void auto_add_state(Automaton *aut,int index,State st){
  if(*aut == NULL){
    *aut = auto_init_automaton();
  }
  (*aut)->states =  auto_state_allocation(aut,index + 1);
  if((*aut)->states[index] != NULL){
    auto_free_state(&((*aut)->states[index]));
  }
  else{
    ((*aut)->n_states)++;
  }
  (*aut)->states[index] = st;
}


State auto_get_state(Automaton aut,int index){
  if(aut == NULL) return NULL;
  if(index >= aut->MAX) return NULL;
  return aut->states[index];
}


void auto_set_initial_state(State st){
  st->control |= 1;
}

void auto_set_final_state(State st){
  st->control |= 2;
}

int auto_is_initial_state(State st){
  return (st->control) & 1;
}

int auto_is_final_state(State st){
  return (st->control) & 2;
}


State auto_init_state(){
  State st;

  st = (State)auto_malloc(sizeof(struct state));
  if(st == NULL) util_error("auto_init_state","allocation of a new state");
  st->control = 0;
  st->terminal_trans = NULL;
  st->auxiliary_trans = NULL;
  return st;
}

Transitions auto_init_transitions(){
  Transitions trans = (Transitions)auto_malloc(sizeof(struct transitions));
  if(trans == NULL) util_error("auto_init_transitions","allocation of new transitions");
  trans->n_transitions = 0;
  trans->input_tags = NULL;
  trans->output_tags = NULL;
  trans->weights = NULL;
  trans->arr = NULL;
  return trans;
}


//option = 0 => keys into tab
//option = 1 => values into tab

Tab_i auto_copy_ll_into_tab(Llist ll,int option){
  struct ll_cell *ptr = ll->head;
  Tab_i tab = NULL;
  int n = ll_length(ll);
  int i;

  for(i = 0 ; i < n ; i++){
    if(option){
      tab_i_assign_value(&tab,i,ptr->value);
    }
    else{
      tab_i_assign_value(&tab,i,ptr->key);
    }
    ptr = ptr->next;
  }
  return tab;
}

void auto_copy_transitions(State *st, Llist term_trans,Llist aux_trans){
  if(term_trans){
    (*st)->terminal_trans = auto_init_transitions();
    (*st)->terminal_trans->input_tags = auto_copy_ll_into_tab(term_trans,0);
    (*st)->terminal_trans->arr = auto_copy_ll_into_tab(term_trans,1);
    (*st)->terminal_trans->n_transitions = tab_i_get_n_elements((*st)->terminal_trans->input_tags);
  }
  if(aux_trans){
    (*st)->auxiliary_trans = auto_init_transitions();
    (*st)->auxiliary_trans->input_tags = auto_copy_ll_into_tab(aux_trans,0);
    (*st)->auxiliary_trans->arr = auto_copy_ll_into_tab(aux_trans,1);
    (*st)->auxiliary_trans->n_transitions = tab_i_get_n_elements((*st)->auxiliary_trans->input_tags);
 }
}


//option = 0 => do not show output tags
//option = 1  => show output tags

void auto_print_transitions(Transitions trans,int option){
  int i;
  if(trans == NULL){
    printf("NULL\n");
    return;
  }
  //  printf("n=%d\n",trans->n_transitions);
  for(i = 0 ; i < trans->n_transitions ; i++){
    if(option)
      printf("(%d,%d,%f,%d);",tab_i_get_value(trans->input_tags,i),tab_i_get_value(trans->output_tags,i),tab_d_get_value(trans->weights,i),tab_i_get_value(trans->arr,i));
    else
      printf("(%d,%f,%d);",tab_i_get_value(trans->input_tags,i),tab_d_get_value(trans->weights,i),tab_i_get_value(trans->arr,i));
  }
  printf("\n");
}

void auto_print_state(State st,int option){
  if(st == NULL){
    printf("----\nEmpty State\n----\n");
    return;
  }
  printf("----------------\n");
  if(st->control & 1) printf("Control: Starting state\n");
  if(st->control & 2) printf("Control: Accepting state\n");
  
  printf("terminal_trans:\n");
  auto_print_transitions(st->terminal_trans,option);
  printf("auxiliary_trans:\n");
  auto_print_transitions(st->auxiliary_trans,option);
  printf("----------------\n");
}


//option = 0 => do not show output tags
//option = 1  => show output tags

void auto_print(Automaton aut,int option){
  int i;
  if(aut == NULL){
      printf("EMPTY AUTOMATON\n");
      return;
    }
  printf("****** *****  *****\n");
  printf("length = %d\n",(int)aut->n_states);
  for(i = 0 ; i < (int)aut->n_states ; i++){
    printf("State %d:\n",i);
    auto_print_state(aut->states[i],option);
  }
  printf("***** ****** ******\n");
}

void auto_println(Automaton aut,int option){
  auto_print(aut,option);
}
