#ifndef _XML_FORMAT_FSA_H_
#define _XML_FORMAT_FSA_H_

#include <string>
#include <boost/lexical_cast.hpp>

#include <outilex/xml.h>
#include <outilex/xml-names.h>


namespace detail {

template<typename Transition, typename XmlFormater>
void xml_write_fsa_trans(xmlwriter & writer, const Transition & tr, XmlFormater formater) {
  
  writer.start_element(xmlnames::TRANS_ELEM);
  writer.write_attribute(xmlnames::TO_ATTR, boost::lexical_cast<std::string>(tr.to()));

  formater.write_label(writer, tr.label());

  writer.end_element();
}

template<typename Fsa, typename XmlFormater>
void xml_read_fsa_state(xmlNode * node, Fsa & fsa, int q, XmlFormater formater) {

  char * text = xmlGetConstProp(node, xmlnames::FINAL_ATTR);
  if (text && text[0] == '1') { fsa.set_final(q); }

  int to;
  typename Fsa::label_type label;

  node = node->children;
  while (node) {

    if (xmlStrcmp(node->name, xmlnames::PROP_ELEM) == 0) {

      formater.read_state_data(node->children, fsa.data(q));
      
    } else if (xmlStrcmp(node->name, xmlnames::TRANS_ELEM) == 0) {

      to = lexical_cast<int>(xmlGetConstProp(node, xmlnames::TO_ATTR));
      formater.read_label(node->next, label);

      fsa.add_trans(q, label, to);
    }

    node = node->next;
  }
}

template<typename Fsa, typename XmlFormater>
void xml_write_fsa_state(xmlwriter & writer, const Fsa & fsa, int q, XmlFormater formater) {

  writer.start_element(xmlnames::STATE_ELEM);
  
  if (fsa.final(q)) { writer.write_attribute(xmlnames::FINAL_ATTR, "1"); }
 
  writer.start_element(xmlnames::PROP_ELEM);
  formater.write_state_data(writer, fsa.data(q));
  writer.end_element();

  typename Fsa::const_trans_iterator tr, end;
  for (tr = fsa.trans_begin(q), end = fsa.trans_end(q); tr != end; ++tr) {
    xml_write_fsa_trans(writer, *tr);
  }

  writer.end_element();
}

} // namespace detail


struct DefaultFsaXmlFormater {

  const char * fsa_name() const { return "fsa"; }

  template<typename Data>
  void read_state_data(xmlNode * node, Data & data) { xml_read(node, data); }

  template<typename Data>
  void write_state_data(xmlwriter & writer, const Data & data) { xml_write(writer, data); }

  template<typename Label>
  void read_label(xmlNode * node, Label & label) { xml_read(node, label); }

  template<typename Label>
  void write_label(xmlwriter & writer, const Label & label) { xml_write(writer, label); }
};


template<typename Fsa, typename XmlFormater = DefaultFsaXmlFormater>
void xml_read_fsa(xmlNode * node, Fsa & fsa, XmlFormater formater) {

  fsa.clear();

  char * text = xmlGetConstProp(node, xmlnames::SIZE_ATTR);
  if (text) {
    fsa.reserve(boost::lexical_cast<int>(text));
  }

  int qno = 0;
  node = node->children;
  while (node) {
    if (xmlStrcmp(node->name, xmlnames::STATE_ELEM) == 0) {
      fsa.resize(qno + 1);
      xml_read_fsa_state(node, fsa, formater);
      ++qno;
    }
    node = node->next;
  }
}

template<typename Fsa, typename XmlFormater = DefaultFsaXmlFormater>
void xml_write_fsa(xmlwriter & writer, const Fsa & fsa, XmlFormater formater) {

  int size = fsa.size();

  writer.start_element(formater.fsa_name());
  writer.write_attribute(xmlnames::SIZE_ATTR, boost::lexical_cast<std::string>(size));

  for (int i = 0; i < size; ++i) {
    xml_write_state(writer, fsa, i, formater);
  }
  writer.end_element();
}

#endif
