#ifndef _FSA_MINIMIZE_H_
#define _FSA_MINIMIZE_H_

namespace detail {


template<typename FsaIn, typename FsaOut>
class fsa_minimizer {


public:

  typedef FsaIn fsa_in;
  typedef FsaOut fsa_out;
  typedef typename fsa_in::label_type label_type;
  typedef typename fsa_in::transition transition;

  static bool compare_by_label(const transition * a, const transition * b) {
    return a->label() < b->label();
  }

  typedef std::set<const transition *, bool (*)(const transition *, const transition *)> transitions;

  const fsa_in & ifsa;
  fsa_out & ofsa;

  std::vector<transitions> trans;
  std::vector<int> colors;
  std::vector<int> nuances;


  fsa_minimizer(const fsa_in & ifsa_, fsa_out & ofsa_)
    : ifsa(ifsa_), ofsa(ofsa_), trans(), colors(), nuances() {}

  void run() {
    
    ofsa.clear();
    if (ifsa.empty()) { return; }

    /* initialisation */

    int size = ifsa.size();
    colors.resize(size);
    nuances.resize(size);
    trans.resize(size, transitions(compare_by_label));

    int nbcolor = 1, nbnuance = 1;

    bool final0 = ifsa.final(0);

    for (int q = 0; q < size; ++q) {

      nuances[q] = ((ifsa.final(q) == final0) ? 0 : (nbnuance = 2, 1));

      for (typename fsa_in::const_trans_iterator tr = ifsa.trans_begin(q), end = ifsa.trans_end(q);
           tr != end; ++tr) {
        trans[q].insert(& * tr);  
      }
    }


    /* color partition */

    do {
    
      colors.swap(nuances);
      nbcolor = nbnuance;

      nbnuance = find_nuances();
 
    } while (nbcolor != nbnuance);
 
 
    /* choose a representant state for each color */

    std::vector<int> repr(nbcolor);
    for (int q = 0; q < size; ++q) { repr[colors[q]] = q; }
 

    /* minimal fsa construction */

    ofsa.resize(nbcolor);

    for (int c = 0; c < nbcolor; ++c) {

      ofsa.set_final(c, ifsa.final(repr[c]));
      ofsa.set_data(c, ifsa.data(repr[c]));

      ofsa.trans_reserve(c, ifsa.trans_size(repr[c]));
      for (typename fsa_in::const_trans_iterator tr = ifsa.trans_begin(repr[c]),
           end = ifsa.trans_end(repr[c]); tr != end; ++tr) {

        ofsa.add_trans(c, tr->label(), colors[tr->to()]);
      }
    }
  }

  
  bool same_trans(int q1, int q2) {

    typename transitions::const_iterator it1 = trans[q1].begin(), end1 = trans[q1].end(),
             it2 = trans[q2].begin(), end2 = trans[q2].end();

    while (it1 != end1 && it2 != end2) {
      
      if (colors[(*it1)->to()] != colors[(*it2)->to()]
          || (*it1)->label() != (*it2)->label()) {
        return false;
      }
      ++it1, ++it2;
    }
    return it1 == end1 && it2 == end2;
  }

  int find_nuances() {
 
    int nbnuance = 0, size = ifsa.size();
    int q1, q2;
 
    for (q1 = 0; q1 < size; ++q1) {
 
      for (q2 = colors[q1]; q2 < q1; ++q2) {
      
        if ((colors[q1] == colors[q2]) && (same_trans(q1, q2))) {
          nuances[q1] = nuances[q2];
          break;
        }
      }

      if (q2 == q1) { // nouvelle nuance
        nuances[q1] = nbnuance++;
      }
    }

    return nbnuance;
  }
};


} // namespace detail

template<typename FsaIn, typename FsaOut>
void fsa_minimize(const FsaIn & ifsa, FsaOut & ofsa) {
  detail::fsa_minimizer<FsaIn, FsaOut>(ifsa, ofsa).run();
}

template<typename Fsa>
void fsa_minimize(Fsa & fsa) {
  Fsa res;
  fsa_minimize(fsa, res);
  res.swap(fsa);
}

#endif

