#ifndef _EPSILON_REMOVAL_H_
#define _EPSILON_REMOVAL_H_

#include <vector>
#include <iterator>


#include <outilex/wgb_color.h>
#include <outilex/transitive_closure.h>


namespace detail {

struct no_label_automaton {

  struct transition {
    int to_;
    transition(int to) : to_(to) {}
    int & to() { return to_; }
    int to() const { return to_; }
  };
  typedef std::vector<transition> transitions; 

  struct state {
    bool final;
    transitions trans;
    state() : final(false), trans() {}
  };

  no_label_automaton() : states() {}


  int size() const { return states.size(); }
  void resize(int size) { states.resize(size); }


  std::vector<state> states;
};

template<typename Output>
class no_input_automaton {

public:

  typedef Output output_type;

  struct transition {
    
    int to_;
    output_type out_;
 
    template<typename Transition>
    transition(const Transition & tr) : to_(tr.to()), out_(tr.out()) {}
    transition(int _to, const output_type & _out) : to_(_to), out_(_out) {}

    inline int to() const { return to_; }
    inline const output_type & out() const { return out_; }

    template<typename Trans>
    transition & operator=(const Trans & tr) {
      to_ = tr.to(); out_ = tr.out(); return *this;
    }
  };

 
  typedef std::vector<transition> transitions;
  typedef typename transitions::iterator trans_iterator;
  typedef typename transitions::const_iterator const_trans_iterator;

  struct state {

    state() : trans() {}

    transitions trans;
  };

  std::vector<state> states;

  inline void resize(int s) { states.resize(s); }
  inline int size() const { return states.size(); }

  inline state & operator[](int i) { return states[i]; }
  inline const state & operator[](int i) const { return states[i]; }


  std::back_insert_iterator<transitions> trans_inserter(int q) {
    return std::back_inserter(states[q].trans);
  }


  inline const_trans_iterator trans_begin(int q) { return states[q].trans.begin(); }
  inline const_trans_iterator trans_end(int q) { return states[q].trans.end(); }


  static const output_type & get_trans_output(const transition & tr) { return tr.out(); }

  void transitive_closure() {

    std::set<output_type> tab[size()];
    wgb_color color[size()];

    for (int i = 0; i < size(); ++i) { color[i] = white; }

    for (int i = 0; i < size(); ++i) {
 
      /* compute transitive */
      transitive_closure(i, output_type::one(), tab, color);

      states[i].trans.clear();
      for (int j = 0; j < size(); ++j) {
        if (i != j) {
          for (typename std::set<output_type>::iterator it = tab[j].begin(); it != tab[j].end(); ++it) {
            states[i].trans.push_back(transition(j, *it));
          }
        }
        tab[j].clear();
      }
    }
  }

protected:

  void transitive_closure(int q, const output_type & curr, std::set<output_type> * res, wgb_color * color) {
  
    if (color[q] == gray) { return; } // avoid infinite recursion
    color[q] = gray;

    res[q].insert(curr);
    for (const_trans_iterator tr = trans_begin(q); tr != trans_end(q); ++tr) {
      transitive_closure(tr->to(), output_type::mult(curr, tr->out()), res, color);
    }
    color[q] = white;
  }
};


template<typename InputIterator, typename OutputIterator1, typename OutputIterator2,
  typename Predicate>
void dispatch_transitions(const Predicate & is_epsilon,
                          InputIterator first, InputIterator last, OutputIterator1 eout,
                          OutputIterator2 nout) {

  while (first != last) {
    if (is_epsilon(first->in())) {
      *eout = *first; ++eout;
    } else {
      *nout = *first; ++nout;
    }
    ++first;
  }
}

}; // namespace detail



template<typename Fst>
struct fst_transition_insert_iterator {

  typedef Fst fst_type;

  fst_type & fst;
  int from;

  inline fst_transition_insert_iterator(Fst & fst_, int q) : fst(fst_), from(q) {}

  inline fst_transition_insert_iterator & operator*() { return *this; }

  template<typename Trans>
  inline void operator=(Trans & tr) { fst.add_trans(from, tr.in(), tr.out(), tr.to()); }

  inline fst_transition_insert_iterator & operator++() { return *this; }
};

template<typename Fst>
inline fst_transition_insert_iterator<Fst> fst_transition_inserter(Fst & fst, int q) {
  return fst_transition_insert_iterator<Fst>(fst, q);
}



template<typename FstIn, typename FstOut, typename IsEpsilonF>
void fst_remove_epsilon(const FstIn & in, FstOut & out, const IsEpsilonF & is_epsilon) {

  typedef FstIn fst_in;
  typedef FstOut fst_out;

  typedef typename fst_in::output_type output_type;
  typedef detail::no_input_automaton<output_type> epsilon_automaton;

  epsilon_automaton Me;

  int size = in.size();

  out.resize(size);
  Me.resize(size);

  for (int q = 0; q < size; q++) {

    // std::cerr << "q=" << q << endl;

    detail::dispatch_transitions(is_epsilon, in.trans_begin(q), in.trans_end(q),
                                 Me.trans_inserter(q), fst_transition_inserter(out, q));
 
    if (in.final(q)) { 
      out.set_final(q);
      out.final_outputs(q) = in.final_outputs(q);
    }
  }


  Me.transitive_closure();

  for (int q = 0; q < size; ++q) {

    for (typename epsilon_automaton::const_trans_iterator etr = Me.trans_begin(q);
         etr != Me.trans_end(q); ++etr) {
    
      if (out.final(etr->to())) {
        out.set_final(q);
        out.final_outputs(q).insert(etr->out());
      }

      for (typename fst_in::const_trans_iterator itr = in.trans_begin(etr->to());
           itr != in.trans_end(etr->to()); ++itr) {

        if (! is_epsilon(itr->in())) {
          out.add_trans(q, itr->in(), output_type::mult(etr->out(), itr->out()), itr->to()); 
        }
      }
    }
  }
}

template<typename FST, typename IsEpsilonF>
void fst_remove_epsilon(FST & fst, const IsEpsilonF & is_epsilon) {
  FST res;
  fst_remove_epsilon(fst, res, is_epsilon);
  fst.swap(res);
}


#if 0
template<typename FstIn, typename FstOut, typename IsEpsilonF>
void synt_fst_remove_epsilon(const FstIn & in, FstOut & out, const IsEpsilonF & is_epsilon) {

  typedef FstIn fst_in;
  typedef FstOut fst_out;

  typedef typename fst_in::output_type output_type;
  typedef detail::no_input_automaton<output_type> epsilon_automaton;

  out.clear(in);

  epsilon_automaton Me;

  int size = in.size();

  out.resize(size);
  Me.resize(size);

  for (int q = 0; q < size; q++) {

    detail::dispatch_transitions(is_epsilon, in.trans_begin(q), in.trans_end(q),
                                 Me.trans_inserter(q), fst_transition_inserter(out, q));
 
    for (typename fst_in::const_synt_trans_iterator itr = in.synt_trans_begin(q);
         itr != in.synt_trans_end(q); ++itr) {

      out.add_synt_trans(q, itr->in(), itr->out(), itr->to()); 
    }
    if (in.final(q)) { out.set_final(q); }
  }


  Me.transitive_closure();

  for (int q = 0; q < size; ++q) {

    for (typename epsilon_automaton::const_trans_iterator etr = Me.trans_begin(q);
         etr != Me.trans_end(q); ++etr) {
    
      for (typename fst_in::const_trans_iterator itr = in.trans_begin(etr->to());
           itr != in.trans_end(etr->to()); ++itr) {

        if (! is_epsilon(itr->in())) {
          out.add_trans(q, itr->in(), output_type::mult(etr->out(), itr->out()), itr->to()); 
        }
      }

      for (typename fst_in::const_synt_trans_iterator itr = in.synt_trans_begin(etr->to());
           itr != in.synt_trans_end(etr->to()); ++itr) {

        out.add_synt_trans(q, itr->in(), output_type::mult(etr->out(), itr->out()), itr->to()); 
      }
    }
  }
}
#endif


template<typename FSAin, typename FSAout, typename Predicate>
void fsa_remove_epsilon(const FSAin & in, FSAout & out, Predicate pred) {
  
  typedef FSAin fsa_in;
  typedef FSAout fsa_out;
 
 
  int size = in.size();

  out.clear();
  out.resize(size);

  int tab[size];

  for (int q = 0; q < size; ++q) {
  
    //out[q] = in[q]; // copy meta data
    out.set_data(q, in.data(q)); // copy meta-data

    for (int i = 0; i < size; ++i) { tab[i] = 0; }
    transitive_closure(in, q, tab, pred);
 
    for (int q2 = 0; q2 < size; ++q2) {
 
      if (tab[q2]) {

        for (typename fsa_in::const_trans_iterator tr = in.trans_begin(q2);
             tr != in.trans_end(q2); ++tr) {
          if (! pred(*tr)) {
            out.add_trans(q, tr->label(), tr->to());
          }
        }
        if (in.final(q2)) { out.set_final(q); }
      }
    }
  }
}

template<typename FSA, typename Predicate>
void fsa_remove_epsilon(FSA & fsa, Predicate is_epsilon) {
  FSA res;
  fsa_remove_epsilon(fsa, res, is_epsilon);
  fsa.swap(res);
}


#endif

