#include <map>
#include <vector>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>

#include <outilex/generic_fst.h>
#include <outilex/unitex_grf.h>
#include <outilex/flatten.h>
#include <outilex/fsa-prune.h>
#include <outilex/epsilon_removal.h>
#include <outilex/fsa-determinize.h>
#include <outilex/fsa-minimize.h>


using namespace std;
using namespace boost;
namespace fs = boost::filesystem;


namespace {

bool is_epsilon(const generic_fst::transition & tr) {
  return (tr.in() == "<E>" || tr.in().empty()) && tr.out().empty();
}

void count_final(generic_fst & fst) {
  int nb = 0;
  for (int q = 0; q < fst.size(); ++q) {
    if (fst.final(q)) { nb++; }
  }
  cout << nb << " finals\n";
}

} // anonymous namespace



int main(int argc, char ** argv) try {

  fs::path fstpath;

  if (argc == 1) { cout << "usage :: flaten <grf8>\n"; return 0; }

  if (argc < 2) { cerr << "bad args\n"; exit(1); }

  fstpath = fs::path(argv[1], fs::native);

  generic_fst fst;

  string fname  = fstpath.leaf();

  string::size_type dot = fname.rfind('.');

  //cout << "loading " << fname << "...\n";

  if (fname.substr(dot) == ".grf8") {
  
    unitex_grf grf(fstpath);
    grf_to_generic_fst(grf, fst);

  } else {
    fst.read(fstpath);
  }
  //cout << "done. (" << fst.size() << " states).\n";

  //cout << "flattening ...\n";
  flatten(fst, fstpath.branch_path());
  //cout << "done. (" << fst.size() << " states).\n";

  //cout << "cleaning ...\n";

  //cout << "pruning ...\n";
  fsa_prune(fst);
  //cout << "done. (" << fst.size() << " states).\n";

  ////cout << "epsilon removal ...\n";
  fsa_remove_epsilon(fst, is_epsilon);
  //cout << "done. (" << fst.size() << " states).\n";

  //cout << "determinization ...\n";
  fsa_determinize(fst);
  //cout << "done. (" << fst.size() << " states).\n";

  //cout << "minimization ...\n";
  fsa_minimize(fst);

  //cout << "done. (" << fst.size() << " states).\n";

  fstpath = change_extension(fstpath, ".gfst");
  fst.write(fstpath);

  //cout << "result in " << fstpath.string() << endl;

} catch (exception & e) {
  cerr << "fatal error: " << e.what() << endl;
  exit(1);
}

