#include "edf_scheduler.h"
#include "types.h"
#include "trace.h"
#include "platform_hooks.h"
#include "constants.h"
#include "specialsignatures.h"
#include "specialclasses.h"
#include "threads.h"
#include "realtime_threads.h"
#include "classes.h"
#include "language.h"
#include "configure.h"
#include "interpreter.h"
#include "memory.h"
#include "exceptions.h"
#include "stack.h"
#include "rconsole.h"
#include "monitor_control.h"

#define NO_OWNER 0x00

#define get_stack_frame() ((StackFrame *) (currentThread->currentStackFrame))

/**
 * Thread currently being executed by engine(). Threads exist in an
 * intrinsic circular list.
 */
Thread* currentThread;

/**
 * Priority queue of threads. Entry points at the last thread in the queue.
 */
REFERENCE threads;
Thread **threadQ;

/**
 * Thread id generator, always increasing.
 */
byte gThreadCounter;

/**
 * Current program number, i.e. number of 'main()'s hanging around
 */
byte gProgramNumber;

void update_stack_frame (StackFrame *stackFrame)
{
  stackFrame->stackTop = curStackTop;
  stackFrame->pc = curPc;
}  

void update_registers (StackFrame *stackFrame)
{
  curPc = stackFrame->pc;
  curStackTop = stackFrame->stackTop;
  curLocalsBase = stackFrame->localsBase;
}

/* Turns out inlines aren't really inlined.
inline byte get_thread_id (Object *obj)
{
  return obj->threadId;
}

inline void set_thread_id (Object *obj, byte threadId)
{
  obj->threadId = threadId;
}

inline void inc_monitor_count (Object *obj)
{
  obj->monitorCount++;
}

inline void set_monitor_count (Object *obj, byte count)
{
  obj->monitorCount = count;
}
*/

#define get_thread_id(sync) ((sync)->threadId)
#define set_thread_id(sync,_threadId) ((sync)->threadId = (_threadId))
#define inc_monitor_count(sync) ((sync)->monitorCount++)
#define set_monitor_count(obj,count) ((sync)->monitorCount = (count))

/**
 * Initialise the thread pool. Note we use a Java object so that we can make
 * the pool available to Java.
 **/
void init_threads()
{
  int i;


  threads = ptr2ref(new_primitive_array(T_REFERENCE, NB_MONITOR_CONTROLS));
  threadQ = (Thread **)ref_array(threads);
  Thread **pQ = threadQ;
  gThreadCounter = 0;
  currentThread = JNULL;
  for (i = 0; i<NB_MONITOR_CONTROLS; i++)
  {
    *pQ++ = null;
  }
  memory_base[MEM_THREADS] = (byte *)threadQ;
  memory_base[MEM_IMAGE] = (byte *)installedBinary;
  memory_base[MEM_STATICS] = (byte *)classStaticStateBase;
}

/**
 * Allocate stack frames
 * Allocate ID
 * Insert into run list
 * Mark thread as STARTED
 */
int init_thread (Thread *thread)
{
  /* Create a new thread and allocate the resources for it. When being used
   * with the GC we must take steps to allow the calling instruction to be
   * re-started. So take care of any changes to global state.
   */
  thread->threadId = gThreadCounter + 1;
  
  // Catch the primordial thread
  if (currentThread == null)
    thread->priority = NORM_PRIORITY;
  

  if (thread->state != NEW)
    return throw_exception(illegalStateException);
  // Protected the argument (that may have come from native code), from the GC
  //protectedRef[0] = (Object *)thread;
  protect_obj(thread);
  // Allocate space for stack frames.
  thread->stackFrameArray = ptr2ref (new_primitive_array (T_BYTE, INITIAL_STACK_FRAMES*sizeof(StackFrame)));
  if (thread->stackFrameArray == JNULL)
  {
    //protectedRef[0] = JNULL;
    unprotect_obj(thread);
    return EXEC_RETRY;
  }
    
  // Allocate actual stack storage (INITIAL_STACK_SIZE * 4 bytes)
  thread->stackArray = ptr2ref (new_primitive_array (T_INT, INITIAL_STACK_SIZE));
  //protectedRef[0] = JNULL;
  unprotect_obj(thread);
  if (thread->stackArray == JNULL)
  {
    free_array (ref2obj(thread->stackFrameArray));
    thread->stackFrameArray = JNULL;
    return EXEC_RETRY;    
  }
  
  gThreadCounter++;
  
  #ifdef VERIFY
  assert (is_array (word2obj (thread->stackFrameArray)), THREADS0);
  assert (is_array (word2obj (thread->stackArray)), THREADS1);
  #endif

  thread->stackFrameArraySize = 0;
  thread->state = STARTED;
  if (currentThread == null)
    currentThread = thread;
    
  enqueue_thread(thread);

  return EXEC_CONTINUE;
}

/**
 * Switches to next thread:
 *
 * do
 *   get next thread
 *   if waiting, grab monitor and run
 *   if sleeping and timer expired, run
 *   if DEAD, clean up and use current thread
 *   if started, initialize and run
 * until selected thread can run
 *  
 * @return false iff there are no live threads
 *         to switch to.
 */
 
boolean switch_thread()
{
  Thread *anchorThread, *previousThread, *candidate;
  Thread **pThreadQ;
  boolean nonDaemonRunnable = false;
  StackFrame *stackFrame = null;
  short i;


  if (currentThread != null)
  {
    // Only current threads can die. Tidy up dead threads
    if (currentThread->state == DEAD)
    {
  
      #if REMOVE_DEAD_THREADS
      // This order of deallocation is actually crucial to avoid leaks
      free_array ((Object *) word2ptr (currentThread->stackArray));
      free_array ((Object *) word2ptr (currentThread->stackFrameArray));

      #ifdef SAFE
      currentThread->stackFrameArray = JNULL;
      currentThread->stackArray = JNULL;
      #endif // SAFE
      #endif // REMOVE_DEAD_THREADS
    
//       // Remove thread from queue.
//       dequeue_thread(currentThread);
	  
      if(currentThread-> priority > 10) {
            dequeue_rt_thread(current_is_rt());
      }
      // Remove thread from queue.
      dequeue_thread(currentThread);
    }
    else { // Save context information
      stackFrame = current_stackframe();

  
      update_stack_frame (stackFrame);
    }
  }

  currentThread = null;
  
  // Loop until a frame is found that can be made to run.
  for (i=NB_MONITOR_CONTROLS-1; i >= 0; i--) {
    pThreadQ = &threadQ[i];

    previousThread = anchorThread = *pThreadQ;
    if (!previousThread)
      continue;

    do
    {
      candidate = word2ptr(previousThread->nextThread);

      
      // See if we can move a thread to the running state. Used to not do this if we
      // had already found one, but turns out to be easiest if we are avoiding
      // priority inversion.
      switch (candidate->state)
      {
        case CONDVAR_WAITING:
          // We are waiting to be notified
          if ((candidate->sleepUntil > 0) && (get_sys_time() >= (FOURBYTES) candidate->sleepUntil))
          {
            // We will drop through to mon waiting.
          }
          else if (candidate->interruptState == INTERRUPT_CLEARED)
            break;
          else
            candidate->interruptState = INTERRUPT_GRANTED;
          
          // candidate->state = MON_WAITING;
          // drop through
        case MON_WAITING:
          {
            objSync *sync = candidate->sync;
            byte threadId = get_thread_id(sync);

            if (threadId == NO_OWNER)
            {
              // NOW enter the monitor (guaranteed to succeed)
              enter_monitor(candidate, word2obj(candidate->waitingOn));
              
              // Set the monitor depth to whatever was saved.
              set_monitor_count(sync, candidate->monitorCount);
              
              // Let the thread run.
              candidate->state = RUNNING;
  
              #ifdef SAFE
              candidate->waitingOn = JNULL;
              candidate->sync = JNULL;
              #endif
            }
#if PI_AVOIDANCE
            // Only avoid priority inversion if we don't already have a thread to run.
            else if (currentThread == null)
            {
                //Only use this if we use non-realtime thread or realtime thread with PIP synchronization
                if (candidate->priority<11 || getCurrentSync()==PIP){
                    Thread *pOwner;
                    int j;

                    // Track down who owns this monitor and run them instead.
                    // Could be 'waiting' in a native method, or we could be deadlocked!
find_next:
                    if (candidate->threadId != threadId)
                    {
                        for (j=NB_MONITOR_CONTROLS-1; j >= 0; j--)
                        {
                          pOwner = threadQ[j];
                          if (!pOwner)
                            continue;

                          do {
                            // Remember threadQ[j] is the last thread on the queue
                            pOwner = word2ptr(pOwner->nextThread);
                            if (pOwner->threadId == threadId)
                            {
                                  if (pOwner->state == RUNNING)
                                  {
                                        currentThread = pOwner;
                                        goto done_pi;
                                      }

                                  // if owner is waiting too, iterate down.
                                  if (pOwner->state == MON_WAITING)
                                  {
                                        threadId = get_thread_id(pOwner->sync);
                                        if (threadId != NO_OWNER)
                                              goto find_next;
                                  }
                            }
                          } while (pOwner != threadQ[j]);
                        }
                        // If we got here, we're in trouble, just drop through.
                    }
                }
            }
done_pi:
		break;
	    ;
#endif // PI_AVOIDANCE
          
          }
          break;
        case JOIN:
          {
            Thread *pThread = (Thread *)word2obj(candidate->waitingOn);
            if (pThread->state == DEAD ||
                candidate->interruptState != INTERRUPT_CLEARED ||
                (candidate->sleepUntil > 0 && get_sys_time() >= (FOURBYTES)candidate->sleepUntil))
            {
              candidate->state = RUNNING;
              candidate->waitingOn = JNULL;
              candidate->sleepUntil = 0;
              if (candidate->interruptState != INTERRUPT_CLEARED)
                candidate->interruptState = INTERRUPT_GRANTED;
            }
            break;
          }
        case SLEEPING:
          if (candidate->interruptState != INTERRUPT_CLEARED
              || (get_sys_time() >= (FOURBYTES) candidate->sleepUntil))
          {
            candidate->state = RUNNING;
            if (candidate->interruptState != INTERRUPT_CLEARED)
          	candidate->interruptState = INTERRUPT_GRANTED;
            #ifdef SAFE
  	    candidate->sleepUntil = JNULL;
            #endif // SAFE
          }
          break;

        case START_DELAYED:
          chooseAndActivateThread();
	  if(currentThread == null) {
            if(get_sys_time() >= candidate->sleepUntil) {
              candidate->state = STARTED;
	        schedule_request(REQUEST_SWITCH_THREAD);
	    }
          }
	  break;

        case STARTED:
          if (currentThread == null) {      
            // Put stack ptr at the beginning of the stack so we can push arguments
            // to entry methods. This assumes set_top_word or set_top_ref will
            // be called immediately below.
	    //
	    
            currentThread = candidate;	// Its just easier this way.
            init_sp_pv();
            candidate->state = RUNNING;
            if (candidate == bootThread) {
              /*
              MethodRecord *mRec;
              ClassRecord *classRecord;
              classRecord = get_class_record (get_entry_class (gProgramNumber));
              // Initialize top word with fake parameter for main():
              set_top_ref_cur (JNULL);
              // Push stack frame for main method:
              mRec= find_method (classRecord, main_4_1Ljava_3lang_3String_2_5V);
              dispatch_special (mRec, null);
              // Push another if necessary for the static initializer:
              dispatch_static_initializer (classRecord, curPc);
*/
              execute_program(gProgramNumber);
            } else {
              set_top_ref_cur (ptr2ref (candidate));
              dispatch_virtual ((Object *) candidate, run_4_5V, null);
            }
            // The following is needed because the current stack frame
            // was just created
            stackFrame = current_stackframe();
            update_stack_frame (stackFrame);
          }
          break;
        case SYSTEM_WAITING:
          // Just keep on waiting
        case RUNNING:
          // Its running already
        case DEAD:
          // Dead threads should be handled earlier
        default:
          // ???
          break;
      }

      // Do we now have a thread we want to run?
      // Note we may later decide not to if all non-daemon threads have died        
      if (currentThread == null && candidate->state == RUNNING)
      {
        currentThread = candidate;
        // Move thread to end of queue
        *pThreadQ = candidate;
      }
      
      if (!candidate->daemon)
      {
      	// May or may not be running but it could do at some point
           nonDaemonRunnable = true;
      }

      // Always use the first running thread as the thread
      // Keep looping: cull dead threads, check there's at least one non-daemon thread
      previousThread = candidate;
    } while (candidate != anchorThread);
  } // end for

  if (nonDaemonRunnable)
  {
    // There is at least one non-daemon thread left alive
    if (currentThread != null)
    {
      // If we found a running thread and there is at least one
      // non-daemon thread left somewhere in the queue...
    
      stackFrame = current_stackframe();

    
      update_registers (stackFrame);

    
      if (currentThread->interruptState == INTERRUPT_GRANTED)
        throw_exception(interruptedException);
    }
      
    return true;
  }

  schedule_request(REQUEST_EXIT);
  currentThread = null;
  
  return false;
}

/*
 * Current thread will wait on the specified object, waiting for a 
 * system_notify. Note the thread does not need to own the object,
 * to wait on it. However it will wait to own the monitor for the
 * object once the wait is complete.
 */
void system_wait(Object *obj)
{

  // Indicate the we are waiting for a system notify
  currentThread->state = SYSTEM_WAITING;
  
  // Set the monitor count for when we resume (always 1).
  currentThread->monitorCount = 1;
  
  // Save the object who's monitor we will want back
  currentThread->waitingOn = ptr2ref (obj);
  currentThread->sync = get_sync(obj);
  
  // no time out
  currentThread->sleepUntil = 0;

  // Gotta yield
  schedule_request( REQUEST_SWITCH_THREAD);
}


/*
 * wake up any objects waiting on the passed system object.
 * Note unlike ordinary waits, we do not allow system waits to be interrupted.
 */
void system_notify(Object *obj, const boolean all)
{
  short i;
  Thread *pThread;

  // Find a thread waiting on us and move to WAIT state.
  for (i=NB_MONITOR_CONTROLS-1; i >= 0; i--)
  {
    pThread = threadQ[i];
    if (!pThread)
      continue;
      
    do {
      // Remember threadQ[i] is the last thread on the queue
      pThread = word2ptr(pThread->nextThread);
      if (pThread->state == SYSTEM_WAITING && pThread->waitingOn == ptr2ref (obj))
      {
        pThread->state = MON_WAITING;
        if (!all)
          return;
      }
    } while (pThread != threadQ[i]);
  }
}


/*
 * Current thread owns object's monitor (we hope) and wishes to relinquish
 * it temporarily (by calling Object.wait()).
 */
int monitor_wait(Object *obj, const FOURBYTES time)
{
  objSync *sync = get_sync(obj);
  if (currentThread->threadId != get_thread_id (sync))
    return throw_exception(illegalMonitorStateException);
  
  // Great. We own the monitor which means we can give it up, but
  // indicate that we are listening for notify's.
  currentThread->state = CONDVAR_WAITING;
  
  // Save monitor depth
  currentThread->monitorCount = get_monitor_count(sync);
  
  // Save the object who's monitor we will want back
  currentThread->waitingOn = ptr2ref (obj);
  currentThread->sync = sync;
  // Might be an alarm set too.
  if (time > 0)
    currentThread->sleepUntil = get_sys_time() + time; 	
  else
    currentThread->sleepUntil = 0;

  // Indicate that the object's monitor is now free.
  set_thread_id (sync, NO_OWNER);
  set_monitor_count(sync, 0);
  
  // Gotta yield
  schedule_request( REQUEST_SWITCH_THREAD);
  return EXEC_CONTINUE;
}

/*
 * Current thread owns object's monitor (we hope) and wishes to wake up
 * any other threads waiting on it. (by calling Object.notify()).
 */
int monitor_notify(Object *obj, const boolean all)
{
  objSync *sync = get_sync(obj);
  if (currentThread->threadId != get_thread_id (sync))
    return throw_exception(illegalMonitorStateException);
  
  monitor_notify_unchecked(obj, all);
  return EXEC_CONTINUE;
}

/*
 * wake up any objects waiting on the passed object.
 */
void monitor_notify_unchecked(Object *obj, const boolean all)
{
  short i;
  Thread *pThread;
  // Find a thread waiting on us and move to WAIT state.
  for (i=NB_MONITOR_CONTROLS-1; i >= 0; i--)
  {
    pThread = threadQ[i];
    if (!pThread)
      continue;
      
    do {
      // Remember threadQ[i] is the last thread on the queue
      pThread = word2ptr(pThread->nextThread);
      if (pThread->state == CONDVAR_WAITING && pThread->waitingOn == ptr2ref (obj))
      {
        // might have been interrupted while waiting
        if (pThread->interruptState != INTERRUPT_CLEARED)
          pThread->interruptState = INTERRUPT_GRANTED;

        pThread->state = MON_WAITING;
        if (!all)
          return;
      }
    } while (pThread != threadQ[i]);
  }
}

/**
 * currentThread enters obj's monitor:
 *
 * if monitor is in use, save object in thread and re-schedule
 * else grab monitor and increment its count.
 * 
 * Note that this operation is atomic as far as the program is concerned.
 */
void enter_monitor (Thread *pThread, Object* obj)
{
  objSync *sync;

  if (obj == JNULL)
  {
    throw_exception (nullPointerException);
    return;
  }
  sync = get_sync(obj);

  //Adding PCE synchronization

  if (/*getCurrentSync() == PCE &&*/ pThread->priority >10){
      MonitorControl *pMC = find_monitor_in_queue(&(obj->sync));
      if (pMC==null && getCurrentSync() == PCE){
          if (pThread->priority > getCurrentDefaultCeilingPriority()){
              MonitorControl* final = getInitialThreadPriority(pThread);
              if (final==null || final->threadPriority >getCurrentDefaultCeilingPriority()){
                throw_exception(ceilingViolationException);
                return;
              }
          }
          pMC = (MonitorControl*) new_object_for_class(LEJOS_REALTIME_MONITORCONTROL);
          pMC->ceilingPriority=getCurrentDefaultCeilingPriority();
          pMC->synchronizationType=getCurrentSync();
          pMC->sync = &(obj->sync);
          pMC->threadPriority = pThread->priority;
          pMC->immortal=0;
          pMC->threadID = pThread->threadId;
          //Adding this condition to avoid getting a lower priority because of nested synchronized blocks
          if (pThread->priority < getCurrentDefaultCeilingPriority()){
              set_thread_priority(pThread,pMC->ceilingPriority);
          }
          else {
              pMC->ceilingPriority = pThread->priority;
          }
          enqueue_mc(pMC);
      }
      else if (pMC!=null && pMC->synchronizationType==PCE) {
          pMC->threadID = pThread->threadId;
          pMC->threadPriority = pThread->priority;
          if (pThread->priority > pMC->ceilingPriority){
              MonitorControl* final = getInitialThreadPriority(pThread);
              if (final == null){
                  printf("final null ! prio %d\n",pThread->priority);
              }
              if (final==null || final->threadPriority >pMC->ceilingPriority){
                throw_exception(ceilingViolationException);
                return;
              }
          }
          //Adding this condition to avoid getting a lower priority because of nested synchronized blocks
          if (pThread->priority < pMC->ceilingPriority){
              set_thread_priority(pThread,pMC->ceilingPriority);
          }
          else {
              pMC->ceilingPriority = pThread->priority;
          }
      }
  }

  if (get_monitor_count (sync) != NO_OWNER && pThread->threadId != get_thread_id (sync))
  {
    // There is an owner, but its not us.
    // Make thread wait until the monitor is relinquished.
    pThread->state = MON_WAITING;
    pThread->waitingOn = ptr2ref (obj);
    pThread->sync = sync;
    pThread->monitorCount = 1;
    // Gotta yield
    schedule_request (REQUEST_SWITCH_THREAD);    
    return;
  }
  set_thread_id (sync, pThread->threadId);
  inc_monitor_count (sync);
}

/**
 * Decrement monitor count
 * Release monitor if count reaches zero
 */
void exit_monitor (Thread *pThread, Object* obj)
{
  byte newMonitorCount;
  objSync *sync;
  if (obj == JNULL)
  {
    // Exiting due to a NPE on monitor_enter [FIX THIS]
    return;
  }

  sync = get_sync(obj);
  #ifdef VERIFY
  assert (get_thread_id(sync) == pThread->threadId, THREADS7);
  assert (get_monitor_count(sync) > 0, THREADS8);
  #endif

  newMonitorCount = get_monitor_count(sync)-1;
  if (newMonitorCount == 0)
    set_thread_id (sync, NO_OWNER);
  set_monitor_count (sync, newMonitorCount);

  //Adding PCE synchronization

  if (/*getCurrentSync() == PCE && */pThread->priority >10){
    MonitorControl *pMC = find_monitor_in_queue(&(obj->sync));

    if (pMC == null || pMC->synchronizationType==PIP){
        return;
    }

    set_thread_priority(pThread,pMC->threadPriority);
    if (pMC->immortal == 0){
        dequeue_mc(pMC);
    }
    else{
        pMC->threadPriority=0;
    }

    schedule_request( REQUEST_SWITCH_THREAD);

  }
}

/**
 * Current thread waits for thread to die.
 *
 * throws InterruptedException
 */
void join_thread(Thread *thread, const FOURBYTES time)
{
  // Make a note of the thread we are waiting for...
  currentThread->waitingOn = ptr2ref (thread);
  // Might be an alarm set too.
  if (time > 0)
    currentThread->sleepUntil = get_sys_time() + time; 	
  else
    currentThread->sleepUntil = 0;
  // Change our state
  currentThread->state = JOIN;
  // Gotta yield
  schedule_request (REQUEST_SWITCH_THREAD);    
}

void dequeue_thread(Thread *thread)
{
  // First take it out of its current queue
  byte cIndex = thread->priority-1;
  Thread **pThreadQ = &threadQ[cIndex];
  
  // Find the previous thread at the old priority
  Thread *previous = *pThreadQ;
  while (word2ptr(previous->nextThread) != thread)
    previous = word2ptr(previous->nextThread);

  if (previous == thread)
  {
    *pThreadQ = null;
  }
  else
  {
    previous->nextThread = thread->nextThread;
    *pThreadQ = previous;
  }  
  chooseAndActivateThread();
}

void enqueue_thread(Thread *thread)
{
  // Could insert it anywhere. Just insert it at the end.
  byte cIndex = thread->priority-1;
  Thread *previous = threadQ[cIndex];
  threadQ[cIndex] = thread;
  if (previous == null)
    thread->nextThread = ptr2ref(thread);
  else {
    Thread *pNext = word2ptr(previous->nextThread);
    thread->nextThread = ptr2ref(pNext);
    previous->nextThread = ptr2ref(thread);
  }
  chooseAndActivateThread();
}

/**
 * Set the priority of the passed thread. Insert into new queue, remove
 * from old queue. Overload to remove from all queues if passed priority
 * is zero.
 *
 * Returns the 'previous' thread.
 */
void set_thread_priority(Thread *thread, const FOURBYTES priority)
{

  if (thread->priority == priority)
    return;

  if (thread->state == NEW)
  {
  	// Not fully initialized
  	thread->priority = priority;
  	return;
  }

  dequeue_thread(thread);
  thread->priority = priority;
  enqueue_thread(thread);      
}

/**
 * Suspend the specified thread. If thread is null suspend all threads
 * except currentThread.
 */
void suspend_thread(Thread *thread)
{
  int i;
  Thread *pThread;
  if (thread)
    thread->state |= SUSPENDED;
  else
  {
    // Suspend all threads
    for (i=NB_MONITOR_CONTROLS-1; i >= 0; i--)
    {
      pThread = threadQ[i];
      if (!pThread)
        continue;
      
      do {
        // Remember threadQ[i] is the last thread on the queue
        pThread = word2ptr(pThread->nextThread);
        if (pThread != currentThread) pThread->state |= SUSPENDED;
      } while (pThread != threadQ[i]);
    }
  }
  schedule_request( REQUEST_SWITCH_THREAD);
}

void resume_thread(Thread *thread)
{
  int i;
  Thread *pThread;
  if (thread)
  {
    thread->state &= ~SUSPENDED;
    return;
  }
  // Suspend all threads
  for (i=NB_MONITOR_CONTROLS-1; i >= 0; i--)
  {
    pThread = threadQ[i];
    if (!pThread)
      continue;
      
    do {
      // Remember threadQ[i] is the last thread on the queue
      pThread = word2ptr(pThread->nextThread);
      pThread->state &= ~SUSPENDED;
    } while (pThread != threadQ[i]);
  }
}
