package lejos.realtime;

import lejos.nxt.comm.RConsole;



/**
 *
 * @author Team iRboT
 */
public abstract class Scheduler {


    protected final static int FP = 1;
    protected final static int EDF = 2;


    protected final static native void setCurrentScheduler(int policyScheduler);
    protected final static native int getCurrentScheduler();

    protected Scheduler(){

    }

    public static Scheduler getDefaultScheduler(){
        int policy = getCurrentScheduler();
        if (policy == FP){
            return PriorityScheduler.instance();
        }
        else if (policy == EDF){
            return EDFScheduler.instance();
        }else{
            throw new NoSuchFieldError("Policy scheduler not found");
        }
    }

    public static void setDefaultScheduler(Scheduler scheduler){
        /*
         Sets the default scheduler. This is the scheduler given to instances of schedulable objects when they are constructed by a Java thread. The default scheduler is set to the required PriorityScheduler at startup.
        Parameters:
        scheduler - The Scheduler that becomes the default scheduler assigned to new schedulable objects created by Java threads. If null nothing happens.
        Throws:
        java.lang.SecurityException - Thrown if the caller is not permitted to set the default scheduler.
         */

        if (scheduler.getPolicyName().equals("EDF")){
            setCurrentScheduler(EDF);
        }
        else if (scheduler.getPolicyName().equals("PriorityScheduler")){
            setCurrentScheduler(FP);
        }else{
            throw new IllegalArgumentException("Scheduler Policy doesn't exist");
        }

    }

    protected abstract boolean addToFeasibility(Schedulable schedulable);
        /*
         Inform this scheduler and cooperating facilities that the resource demands of the given instance of Schedulable will be considered in the feasibility analysis of the associated Scheduler until further notice. Whether the resulting system is feasible or not, the addition is completed.
        If the object is already included in the feasibility set, do nothing.

        Parameters:
        schedulable - A reference to the given instance of Schedulable
        Returns:
        True, if the system is feasible after the addition. False, if not.
        Throws:
        java.lang.IllegalArgumentException - Thrown if schedulable is null, or if schedulable is not associated with this; that is schedulable.getScheduler() != this.
        */


    public abstract void fireSchedulable(Schedulable schedulable);
        /*
         Trigger the execution of a schedulable object (like an AsyncEventHandler).
        Parameters:
        schedulable - The schedulable object to make active. If null, nothing happens.
        Throws:
        java.lang.UnsupportedOperationException - Thrown if the scheduler cannot release schedulable for execution.
        */

    public abstract java.lang.String getPolicyName();
        /*
         Gets a string representing the policy of this. The string value need not be interned, but it must be created in a memory area that does not cause an illegal assignment error if stored in the current allocation context and does not cause a MemoryAccessError when accessed.
        Returns:
        A String object which is the name of the scheduling policy used by this.
         */

    public abstract boolean isFeasible();
        /*
         Queries the system about the feasibility of the system currently being considered. The definitions of "feasible" and "system" are the responsibility of the feasibility algorithm of the actual Scheduler subclass.
        Returns:
        True, if the system is feasible. False, if not.
         */

    protected abstract boolean removeFromFeasibility(Schedulable schedulable);
        /*
         Inform this scheduler and cooperating facilities that the resource demands of the given instance of Schedulable should no longer be considered in the feasibility analysis of the associated Scheduler. Whether the resulting system is feasible or not, the removal is completed.
        Parameters:
        schedulable - A reference to the given instance of Schedulable
        Returns:
        True, if the removal was successful. False, if the schedulable object cannot be removed from the scheduler's feasibility set; e.g., the schedulable object is not part of the scheduler's feasibility set.
        Throws:
        java.lang.IllegalArgumentException - Thrown if schedulable is null.
         */

    public abstract boolean setIfFeasible(Schedulable schedulable,ReleaseParameters release, MemoryParameters memory);
        /*
         This method first performs a feasibility analysis using the proposed parameter objects as replacements for the current parameters of schedulable. If the resulting system is feasible, this method replaces the current parameters of schedulable with the proposed ones.
        This method does not require that the schedulable object be in the feasibility set before it is called. If it is not initially a member of the feasibility set it will be added if the resulting system is feasible.

        Parameters:
        schedulable - The schedulable object for which the changes are proposed.
        release - The proposed release parameters. If null, the default value of this scheduler is used (a new object is created if the default value is not null). (See PriorityScheduler.)
        memory - The proposed memory parameters. If null, the default value of this scheduler is used (a new object is created if the default value is not null). (See PriorityScheduler.)
        Returns:
        True, if the resulting system is feasible and the changes are made. False, if the resulting system is not feasible and no changes are made.
        Throws:
        java.lang.IllegalArgumentException - Thrown if schedulable is null, or schedulable is not associated with this scheduler, or the proposed parameters are not compatible with this scheduler.
        IllegalAssignmentError - Thrown if schedulable cannot hold references to the proposed parameter objects, or the parameter objects cannot hold a reference to schedulable.
        java.lang.IllegalThreadStateException - Thrown if the new release parameters change schedulable from periodic scheduling to some other protocol and schedulable is currently waiting for the next release in RealtimeThread.waitForNextPeriod() or RealtimeThread.waitForNextPeriodInterruptible().
        */

    public abstract boolean setIfFeasible(Schedulable schedulable, ReleaseParameters release, MemoryParameters memory, ProcessingGroupParameters group);
        /*
         This method first performs a feasibility analysis using the proposed parameter objects as replacements for the current parameters of schedulable. If the resulting system is feasible, this method replaces the current parameters of schedulable with the proposed ones.
        This method does not require that the schedulable object be in the feasibility set before it is called. If it is not initially a member of the feasibility set it will be added if the resulting system is feasible.

        Parameters:
        schedulable - The schedulable object for which the changes are proposed.
        release - The proposed release parameters. If null, the default value of this scheduler is used (a new object is created if the default value is not null). (See PriorityScheduler.)
        memory - The proposed memory parameters. If null, the default value of this scheduler is used (a new object is created if the default value is not null). (See PriorityScheduler.)
        group - The proposed processing group parameters. If null, the default value of this scheduler is used (a new object is created if the default value is not null). (See PriorityScheduler.)
        Returns:
        True, if the resulting system is feasible and the changes are made. False, if the resulting system is not feasible and no changes are made.
        Throws:
        java.lang.IllegalArgumentException - Thrown if schedulable is null, or schedulable is not associated with this scheduler, or the proposed parameters are not compatible with this scheduler.
        IllegalAssignmentError - Thrown if schedulable cannot hold references to the proposed parameter objects, or the parameter objects cannot hold a reference to schedulable.
        java.lang.IllegalThreadStateException - Thrown if the new release parameters change schedulable from periodic scheduling to some other protocol and schedulable is currently waiting for the next release in RealtimeThread.waitForNextPeriod() or RealtimeThread.waitForNextPeriodInterruptible().
        */


    public abstract boolean setIfFeasible(Schedulable schedulable, SchedulingParameters scheduling, ReleaseParameters release, MemoryParameters memory, ProcessingGroupParameters group);
        /*
         This method first performs a feasibility analysis using the proposed parameter objects as replacements for the current parameters of schedulable. If the resulting system is feasible, this method replaces the current parameters of schedulable with the proposed ones.
        This method does not require that the schedulable object be in the feasibility set before it is called. If it is not initially a member of the feasibility set it will be added if the resulting system is feasible.

        Parameters:
        schedulable - The schedulable object for which the changes are proposed.
        scheduling - The proposed scheduling parameters. If null, the default value of this scheduler is used (a new object is created if the default value is not null). (See PriorityScheduler.)
        release - The proposed release parameters. If null, the default value of this scheduler is used (a new object is created if the default value is not null). (See PriorityScheduler.)
        memory - The proposed memory parameters. If null, the default value of this scheduler is used (a new object is created if the default value is not null). (See PriorityScheduler.)
        group - The proposed processing group parameters. If null, the default value of this scheduler is used (a new object is created if the default value is not null). (See PriorityScheduler.)
        Returns:
        True, if the resulting system is feasible and the changes are made. False, if the resulting system is not feasible and no changes are made.
        Throws:
        java.lang.IllegalArgumentException - Thrown if schedulable is null, or schedulable is not associated with this scheduler, or the proposed parameters are not compatible with this scheduler.
        IllegalAssignmentError - Thrown if schedulable cannot hold references to the proposed parameter objects, or the parameter objects cannot hold a reference to schedulable.
        java.lang.IllegalThreadStateException - Thrown if the new release parameters change schedulable from periodic scheduling to some other protocol and schedulable is currently waiting for the next release in RealtimeThread.waitForNextPeriod() or RealtimeThread.waitForNextPeriodInterruptible().
        */

}
