package lejos.realtime;

/**
 * Handlers and other objects can be run by a Scheduler if they provide a
 * {@code run() } method and the methods defined below. The Scheduler uses this
 * information to create a suitable context to execute the {@code run() }
 * method.
 *
 * @author Benoit Bousquet
 */
public interface Schedulable extends Runnable {

    /**
     * Sets the release parameters associated with this instance of
     * {@code Schedulable}.
     *
     * Since this affects the constraints expressed in the release parameters
     * of the existing schedulable objects, this may change the feasibility of
     * the current system.
     *
     * This change becomes effective under conditions determined by the
     * scheduler controlling the schedulable object. For instance, the change
     * may be immediate or it may be delayed until the next release of the
     * schedulable object. The different properties of the release parameters
     * may take effect at different times. See the documentation for the
     * scheduler for details.
     * 
     * @param release
     *          A {@link ReleaseParameters} object which will become the release
     * parameters associated with this after the method call, and take effect
     * as determined by the associated scheduler. If null, the default value is
     * governed by the associated scheduler (a new object is created if the
     * default value is not null). (See {@link PriorityScheduler}.)
     *
     * @throws IllegalArgumentException
     *          Never throwed for the moment.
     *
     * @throws IllegalThreadStateException
     *          Never throwed for the moment.
     */
    public void setReleaseParameters(ReleaseParameters release) throws IllegalArgumentException, IllegalStateException;

    /**
     * Sets the scheduling parameters associated with this instance of
     * Schedulable.
     *
     * Since this affects the scheduling parameters of the existing schedulable
     * objects, this may change the feasibility of the current system.
     *
     * This change becomes effective under conditions determined by the
     * scheduler controlling the schedulable object. For instance, the change
     * may be immediate or it may be delayed until the next release of the
     * schedulable object. See the documentation for the scheduler for details.
     * 
     * @param scheduling
     *          A reference to the {@link SchedulingParameters} object. If null,
     * the default value is governed by the associated scheduler (a new object
     * is created if the default value is not null).
     * (See {@link PriorityScheduler.)}
     *
     * @throws IllegalArgumentException
     *          Never throwed for the moment.
     *
     * @throws IllegalAssignmentError
     *          Never throwed for the moment.
     */
    public void setSchedulingParameters(SchedulingParameters scheduling) throws IllegalArgumentException, IllegalAssignmentError;

    boolean addIfFeasible();

   /**
    * This method first performs a feasibility analysis with this added to the system.
    */

    boolean addToFeasibility();
/*          Inform the scheduler and cooperating facilities that this instance of Schedulable should be considered in feasibility analysis until further notified.*/

    MemoryParameters getMemoryParameters();
 /*         Gets a reference to the MemoryParameters object for this schedulable object.*/

    ProcessingGroupParameters getProcessingGroupParameters();
/*          Gets a reference to the ProcessingGroupParameters object for this schedulable object.*/

    ReleaseParameters getReleaseParameters();
/*          Gets a reference to the ReleaseParameters object for this schedulable object.*/

    Scheduler getScheduler();
/*          Gets a reference to the Scheduler object for this schedulable object.*/

    SchedulingParameters getSchedulingParameters();
/*          Gets a reference to the SchedulingParameters object for this schedulable object.*/

    boolean removeFromFeasibility() ;
/*          Inform the scheduler and cooperating facilities that this instance of Schedulable should not be considered in feasibility analysis until it is further notified.*/

   boolean setIfFeasible(ReleaseParameters release, MemoryParameters memory);

    /*      This method first performs a feasibility analysis using the proposed parameter objects as replacements for the current parameters of this.*/

    boolean setIfFeasible(ReleaseParameters release, MemoryParameters memory, ProcessingGroupParameters group) ;
/*          This method first performs a feasibility analysis using the proposed parameter objects as replacements for the current parameters of this.*/

    boolean setIfFeasible(ReleaseParameters release, ProcessingGroupParameters group) ;
/*          This method first performs a feasibility analysis using the proposed parameter objects as replacements for the current parameters of this.*/

    boolean setIfFeasible(SchedulingParameters scheduling, ReleaseParameters release, MemoryParameters memory) ;
/*          This method first performs a feasibility analysis using the proposed parameter objects as replacements for the current parameters of this.*/

    boolean setIfFeasible(SchedulingParameters scheduling, ReleaseParameters release, MemoryParameters memory, ProcessingGroupParameters group) ;
/*          This method first performs a feasibility analysis using the proposed parameter objects as replacements for the current parameters of this.*/

    void setMemoryParameters(MemoryParameters memory) ;
/*          Sets the memory parameters associated with this instance of Schedulable.*/

    boolean setMemoryParametersIfFeasible(MemoryParameters memory) ;
/*          This method first performs a feasibility analysis using the proposed parameter object as replacement for the current parameter of this.*/

    void setProcessingGroupParameters(ProcessingGroupParameters group) ;
/*          Sets the ProcessingGroupParameters of this.*/

    boolean setProcessingGroupParametersIfFeasible(ProcessingGroupParameters group) ;
/*          This method first performs a feasibility analysis using the proposed parameter object as replacement for the current parameter of this.*/

    boolean setReleaseParametersIfFeasible(ReleaseParameters release);
 /*         This method first performs a feasibility analysis using the proposed parameter object as replacement for the current parameter of this.*/

    void setScheduler(Scheduler scheduler);
/*          Sets the reference to the Scheduler object.*/

     void setScheduler(Scheduler scheduler, SchedulingParameters scheduling, ReleaseParameters release, MemoryParameters memoryParameters, ProcessingGroupParameters group) ;
/*          Sets the scheduler and associated parameter objects.*/

    boolean setSchedulingParametersIfFeasible(SchedulingParameters scheduling);
/*          This method first performs a feasibility analysis using the proposed parameter object as replacement for the current parameter of this.*/


}
